/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.export;

import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * This action set the MCC file name as attachment in the header
 */
public class SetMCCHeaderAction extends AbstractSetContentDispositionHeaderAction implements Serviceable
{
    /** Version to suffix */
    public static final Map<String, String> VERSION_TO_SUFFIX = Map.of(
        "final", StringUtils.EMPTY,
        "work", "-complet",
        "diff", "-modifications"
    );
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _catalogsManager = (CatalogsManager) serviceManager.lookup(CatalogsManager.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    protected String buildFilename(Request request, Parameters parameters) throws Exception
    {
        String programItemId = request.getParameter("programItem");
        ProgramItem programItem = _resolver.resolveById(programItemId);
        String version = parameters.getParameter("version");
        String format = parameters.getParameter("format");
        String prefix = parameters.getParameter("prefix", StringUtils.EMPTY);
        
        StringBuilder filename = new StringBuilder();
        filename.append(prefix);
        filename.append(_catalogsManager.getCatalog(programItem.getCatalog()).getTitle());
        filename.append("-");
        filename.append(((Content) programItem).getTitle());
        filename.append("-");
        filename.append(programItem.getCode());
        filename.append(VERSION_TO_SUFFIX.getOrDefault(version, "-" + version));
        filename.append(".");
        filename.append(format);
        
        return filename.toString();
    }
}
