/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.right;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.rights.ContentAccessController;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessController.Permission.PermissionType;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.rule.ThematicsHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Access controller for thematics.
 */
public class ThematicAccessController implements AccessController, Serviceable, PluginAware
{
    private static final String __ALL_THEMATIC_CONTENT_FAKE_CONTEXT = "$allThematicContent$";

    private static List<String> _RIGHT_IDS = List.of("Workflow_Rights_Edition_Online");
    
    private ContentTypesHelper _contentTypesHelper;
    private RightManager _rightManager;
    private ThematicsHelper _thematicHelper;

    private String _id;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _thematicHelper = (ThematicsHelper) manager.lookup(ThematicsHelper.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    @Override
    public boolean isSupported(Object object)
    {
        return object instanceof Content content && _contentTypesHelper.isInstanceOf(content, RulesManager.THEMATIC_CONTENT_TYPE);
    }
    
    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        // FIXME CMS-11814 ContentTypeAccessController while reloading cache in web context
        return AccessResult.ANY_CONNECTED_ALLOWED;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        // FIXME CMS-11814 ContentTypeAccessController while reloading cache in web context
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        // FIXME CMS-11814 ContentTypeAccessController while reloading cache in web context
        return AccessResult.ANY_CONNECTED_ALLOWED;
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        // FIXME CMS-11814 ContentTypeAccessController while reloading cache in web context
        return Collections.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        // FIXME CMS-11814 ContentTypeAccessController while reloading cache in web context
        return Collections.EMPTY_MAP;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (_RIGHT_IDS.contains(rightId))
        {
            return _thematicHelper.hasHandleThematicRight(user) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
        }
        
        return AccessResult.UNKNOWN;
    }

    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Map<String, AccessResult> permissionByRight = new HashMap<>();
        
        if (_thematicHelper.hasHandleThematicRight(user))
        {
            _RIGHT_IDS.stream()
                .forEach(r -> {
                    permissionByRight.put(r, AccessResult.USER_ALLOWED);
                });
        }
        
        return permissionByRight;
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        Map<UserIdentity, AccessResult> permissionByUser = new HashMap<>();
        
        if (_RIGHT_IDS.contains(rightId))
        {
            for (UserIdentity user : _rightManager.getAllowedUsers(ThematicsHelper.HANDLE_THEMATICS_RIGHT, "/${WorkspaceName}").getAllowedUsers())
            {
                permissionByUser.put(user, AccessResult.USER_ALLOWED);
            }
        }
        return permissionByUser;
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Collections.EMPTY_MAP;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }
    
    @Override
    public AccessExplanation getStandardAccessExplanation(AccessResult permission, Object object)
    {
        switch (permission)
        {
            case USER_ALLOWED:
            case ANY_CONNECTED_ALLOWED:
            case UNKNOWN:
                return new AccessExplanation(getId(), permission,
                        new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_RIGHTS_THEMATICS_ACCESS_CONTROLLER_" + permission.name() + "_EXPLANATION",
                                Map.of(
                                        "objectLabel", getObjectLabelForExplanation(object)
                                    )
                                )
                        );
            default:
                return AccessController.getDefaultAccessExplanation(getId(), permission);
        }
    }
    
    private I18nizableText getObjectLabelForExplanation(Object object)
    {
        if (object.equals(__ALL_THEMATIC_CONTENT_FAKE_CONTEXT))
        {
            return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_RIGHTS_THEMATICS_ACCESS_CONTROLLER_ALL_THEMATICS_EXPLANATION_LABEL");
        }
        else if (object instanceof Content content)
        {
            return new I18nizableText(
                    "plugin.odf-pilotage",
                    "PLUGINS_ODF_PILOTAGE_RIGHTS_THEMATICS_ACCESS_CONTROLLER_CONTENT_EXPLANATION_LABEL",
                    Map.of("title", new I18nizableText(content.getTitle()))
                );
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }

    
    @Override
    public I18nizableText getObjectLabel(Object object)
    {
        if (object.equals(__ALL_THEMATIC_CONTENT_FAKE_CONTEXT))
        {
            return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_RIGHTS_THEMATICS_ACCESS_CONTROLLER_ALL_THEMATICS_LABEL");
        }
        return new I18nizableText(((Content) object).getTitle());
    }
    
    public Map<ExplanationObject, Map<Permission, AccessExplanation>> explainAllPermissions(UserIdentity identity, Set<GroupIdentity> groups)
    {
        // Do not list all the content of type thematic.
        // Instead take advantage of the fact that the getPermission methods
        // ignore the context to call them with a fake context to generate a explanation for "all the content of type thematic"
        Map<Permission, AccessExplanation> contextPermissions = new HashMap<>();
        AccessExplanation explanation = explainReadAccessPermission(identity, groups, __ALL_THEMATIC_CONTENT_FAKE_CONTEXT);
        if (explanation.accessResult() != AccessResult.UNKNOWN)
        {
            contextPermissions.put(new Permission(PermissionType.READ, null), explanation);
        }
        
        for (String rightId: _RIGHT_IDS)
        {
            explanation = explainPermission(identity, groups, rightId, __ALL_THEMATIC_CONTENT_FAKE_CONTEXT);
            if (explanation.accessResult() != AccessResult.UNKNOWN)
            {
                contextPermissions.put(new Permission(PermissionType.RIGHT, rightId), explanation);
            }
        }
        
        return Map.of(getExplanationObject(__ALL_THEMATIC_CONTENT_FAKE_CONTEXT), contextPermissions);
    }
    
    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        return ContentAccessController.CONTENT_CONTEXT_CATEGORY;
    }
    
    public int getObjectPriority(Object object)
    {
        if (object.equals(__ALL_THEMATIC_CONTENT_FAKE_CONTEXT))
        {
            return 5;
        }
        return AccessController.super.getObjectPriority(object);
    }
    
    public String getId()
    {
        return _id;
    }
}
