/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.schedulable;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;

import org.apache.cocoon.components.ContextHelper;
import org.quartz.JobDataMap;

import org.ametys.core.schedule.Schedulable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.odfpilotage.report.PilotageReport;
import org.ametys.plugins.odfpilotage.report.PilotageReport.PilotageReportTarget;
import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.workspace.WorkspaceMatcher;

/**
 * {@link Schedulable} for pilotage report by program.
 */
public class ProgramReportSchedulable extends AbstractReportSchedulable
{
    /** The key for the program */
    public static final String JOBDATAMAP_PROGRAM_KEY = "program";
    
    @Override
    public PilotageReportTarget getTarget()
    {
        return PilotageReportTarget.PROGRAM;
    }

    @Override
    protected Map<String, String> getReportParameters(JobDataMap jobDataMap)
    {
        Map<String, String> reportParameters = super.getReportParameters(jobDataMap);
        reportParameters.put(PilotageReport.PARAMETER_PROGRAM, jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_PROGRAM_KEY));
        return reportParameters;
    }
    
    @Override
    // FIXME CMS-9980
    public Map<String, ElementDefinition> getParameters()
    {
        Map<String, ElementDefinition> original = super.getParameters();
        
        boolean isAdminContext = Optional.ofNullable(_context)
            .map(ContextHelper::getRequest)
            .map(r -> r.getAttribute(WorkspaceMatcher.WORKSPACE_NAME))
            .map("admin"::equals)
            .orElse(false);
        
        // Remove the widget on program field if we are in an admin context
        if (isAdminContext && original.containsKey(JOBDATAMAP_PROGRAM_KEY))
        {
            // Copy to avoid the modification of the original object
            Map<String, ElementDefinition> copy = new LinkedHashMap<>(original);
            ElementDefinition<String> originalElement = original.get(JOBDATAMAP_PROGRAM_KEY);
            ElementDefinition<String> copiedElement = new DefaultElementDefinition<>(originalElement);
            copiedElement.setWidget(null);
            copiedElement.setWidgetParameters(Collections.EMPTY_MAP);
            copy.put(JOBDATAMAP_PROGRAM_KEY, copiedElement);
            return copy;
        }
        
        return original;
    }
}
