/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.validator;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.holder.DataHolderDisableConditionsEvaluator;
import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.disableconditions.DisableConditions;
import org.ametys.runtime.model.disableconditions.DisableConditionsEvaluator;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * Global validator for content.
 * Check that every instance of the repeater {@link #getRepeaterName()} have a different value in {@link #getContentDataName()}.
 */
public abstract class AbstractRepeaterWithUniqueContentValidator extends AbstractContentValidator implements Serviceable
{
    /** The {@link DisableConditions} evaluator */
    protected DisableConditionsEvaluator<ModelAwareDataHolder> _disableConditionsEvaluator;
    
    @SuppressWarnings("unchecked")
    public void service(ServiceManager manager) throws ServiceException
    {
        _disableConditionsEvaluator = (DisableConditionsEvaluator<ModelAwareDataHolder>) manager.lookup(DataHolderDisableConditionsEvaluator.ROLE);
    }
    
    @Override
    public ValidationResult validate(Content content)
    {
        ValidationResult result = new ValidationResult();
        
        String repeaterName = getRepeaterName();
        ModelItem repeaterDefinition = content.getDefinition(repeaterName);
        
        if (content.hasValue(repeaterName) && !_disableConditionsEvaluator.evaluateDisableConditions(repeaterDefinition, repeaterName, content))
        {
            Set<String> itemIds = new HashSet<>();
            
            ModelAwareRepeater repeater = content.getRepeater(repeaterName);
            
            String contentDataName = getContentDataName();
            ModelItem contentDataDefinition = content.getDefinition(repeaterName + ModelItem.ITEM_PATH_SEPARATOR + contentDataName);
            
            for (ModelAwareRepeaterEntry entry : repeater.getEntries())
            {
                String dataPath = repeaterName + "[" + entry.getPosition() + "]" + ModelItem.ITEM_PATH_SEPARATOR + contentDataName;
                if (!_disableConditionsEvaluator.evaluateDisableConditions(contentDataDefinition, dataPath, content))
                {
                    ContentValue item = entry.getValue(getContentDataName());
                    
                    if (item != null && !itemIds.add(item.getContentId()))
                    {
                        String title = item.getContentIfExists()
                                .map(Content::getTitle)
                                .orElse(item.getContentId());
                        
                        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
                        i18nParams.put("contentTitle", new I18nizableText(content.getTitle()));
                        i18nParams.put("dataValueTitle", new I18nizableText(title));
                        result.addError(new I18nizableText("plugin.odf-pilotage", getErrorKey(), i18nParams));
                    }
                }
            }
        }
        
        return result;
    }
    
    @Override
    public ValidationResult validate(Content content, Map<String, Object> values, View view)
    {
        ValidationResult result = new ValidationResult();
        
        Object repeater = values.get(getRepeaterName());
        List<Map<String, Object>> repeaterEntries =  repeater instanceof SynchronizableRepeater synchronizableRepeater
                    ? synchronizableRepeater.getEntries()
                    : List.of(); // repeater can be an UntouchedValue, if it is disable or non-writable

        if (!repeaterEntries.isEmpty())
        {
            Set<String> itemIds = new HashSet<>();
            ModelItem modelItem = content.getDefinition(getRepeaterName() + ModelItem.ITEM_PATH_SEPARATOR + getContentDataName());
            
            for (int i = 0; i < repeaterEntries.size(); i++)
            {
                Map<String, Object> entry = repeaterEntries.get(i);
                int position = i + 1;
                
                Optional<String> oldDataPath = Optional.of(repeaterEntries)
                                                       .filter(SynchronizableRepeater.class::isInstance)
                                                       .map(SynchronizableRepeater.class::cast)
                                                       .flatMap(syncRepeater -> syncRepeater.getPreviousPosition(position))
                                                       .map(previousPosition -> getRepeaterName() + "[" + previousPosition + "]" + ModelItem.ITEM_PATH_SEPARATOR + getContentDataName());
                
                Content repeaterContent = Optional.ofNullable(getContentDataName())
                                                  .map(entry::get)
                                                  .map(value -> DataHolderHelper.getValueFromSynchronizableValue(value, content, modelItem, oldDataPath, SynchronizationContext.newInstance()))
                                                  .filter(ContentValue.class::isInstance)
                                                  .map(ContentValue.class::cast)
                                                  .flatMap(ContentValue::getContentIfExists)
                                                  .orElse(null);   // value can be an UntouchedValue, if it is disabled or non-writable
                
                if (repeaterContent != null && !itemIds.add(repeaterContent.getId()))
                {
                    Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
                    i18nParams.put("contentTitle", new I18nizableText(content.getTitle()));
                    i18nParams.put("dataValueTitle", new I18nizableText(repeaterContent.getTitle()));
                    result.addError(new I18nizableText("plugin.odf-pilotage", getErrorKey(), i18nParams));
                }
            }
        }
        
        return result;
    }
    
    /**
     * Get the repeater name.
     * @return The repeater name.
     */
    protected abstract String getRepeaterName();
    
    /**
     * Get the content data name.
     * @return The content data name.
     */
    protected abstract String getContentDataName();
    
    /**
     * Get the error I18N key (parameterizable with "contentTitle" and "dataValueTitle").
     * @return The error 18N key.
     */
    protected abstract String getErrorKey();
}
