/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.observation;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.engine.BackgroundEngineHelper;
import org.ametys.core.engine.BackgroundEnvironment;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Container;
import org.ametys.plugins.core.user.UserDAO;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * Observer to update MCC regime on children.
 */
public class MccRegimeObserver extends AbstractLogEnabled implements Observer, Contextualizable, Serviceable, Initializable
{
    private static final String __ALREADY_EXPLORING = MccRegimeObserver.class.getName() + "$alreadyRunning";

    private static ExecutorService __THREAD_EXECUTOR;
    
    private boolean _isActive;
    private boolean _force;

    private Context _context;
    private org.apache.cocoon.environment.Context _environmentContext;
    
    private ServiceManager _serviceManager;
    private ODFHelper _odfHelper;
    private PilotageHelper _pilotageHelper;
    private ContentWorkflowHelper _contentWorkflowHelper;
    private CurrentUserProvider _currentUserProvider;
    private AmetysObjectResolver _resolver;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
        _environmentContext = (org.apache.cocoon.environment.Context) _context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _serviceManager = smanager;
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _pilotageHelper = (PilotageHelper) smanager.lookup(PilotageHelper.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void initialize() throws Exception
    {
        String regimePolicy = _pilotageHelper.getMCCRegimePolicy();
        _force = regimePolicy.equals("FORCE");
        _isActive = _force || regimePolicy.equals("DEFAULT");
        __THREAD_EXECUTOR = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors(), new AsyncWorkflowCallThreadFactory());
    }
    
    public boolean supports(Event event)
    {
        if (_isActive && event.getId().equals(ObservationConstants.EVENT_CONTENT_MODIFIED))
        {
            return event.getArguments().get(ObservationConstants.ARGS_CONTENT) instanceof ProgramItem;
        }
        
        return false;
    }

    public int getPriority(Event event)
    {
        return 2000;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);

        // There are two ways to launch this observer:
        //  - edit a program item
        //  - this observer itself call the modify action (2) with its courses children
        // and call this observer again with each of it
        //
        // To avoid multiple recursive calls, we check a flag put in a request attribute.
        boolean alreadyExploring = Optional.of(request)
                .map(r -> r.getAttribute(__ALREADY_EXPLORING))
                .map(v -> (boolean) v)
                .orElse(false);
        
        if (!alreadyExploring)
        {
            ProgramItem programItem = (ProgramItem) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            Container stepHolder = _pilotageHelper.getStepHolder(programItem).getRight();
            if (stepHolder != null && stepHolder.hasValue(PilotageHelper.CONTAINER_MCC_REGIME))
            {
                _updateCourses(programItem, stepHolder, stepHolder.getValue(PilotageHelper.CONTAINER_MCC_REGIME));
            }
        }
    }
    
    private void _updateCourses(ProgramItem programItem, Container stepHolder, ContentValue mccRegime)
    {
        // Update mccRegime field
        // If :
        //  - programItem is a course AND
        //  - stepHolder is the step holder of the current course AND
        //      - course doesn't have value or empty value for mccRegime field OR
        //      - force the value AND current value is different than value to set
        if (
                programItem instanceof Course course
                && (
                    !course.hasValueOrEmpty(PilotageHelper.COURSE_MCC_REGIME)
                    || _force && !mccRegime.equals(course.getValue(PilotageHelper.COURSE_MCC_REGIME))
                )
                && stepHolder.equals(_pilotageHelper.getStepHolder(programItem).getRight())
            )
        {
            // Set value manually because of restrictions
            course.setValue(PilotageHelper.COURSE_MCC_REGIME, mccRegime);
            course.saveChanges();
            
            // Call the workflow on the content to return to draft state - standard modify action (id = 2)
            _callWorkflowAsynchronously(course);
        }
        
        // Explore children
        _odfHelper.getChildProgramItems(programItem).forEach(child -> _updateCourses(child, stepHolder, mccRegime));
    }
    
    private void _callWorkflowAsynchronously(Course course)
    {
        __THREAD_EXECUTOR.submit(new UpdateWorkflow(course, _currentUserProvider.getUser(), new SLF4JLoggerAdapter(getLogger())));
    }
    
    private class UpdateWorkflow implements Runnable
    {
        private String _courseId;
        private UserIdentity _user;
        private Logger _logger;
        
        public UpdateWorkflow(Course course, UserIdentity user, Logger logger)
        {
            _courseId = course.getId();
            _user = user;
            _logger = logger;
        }
        
        @Override
        public void run()
        {
            Map<String, Object> environmentInformation = BackgroundEngineHelper.createAndEnterEngineEnvironment(_serviceManager, _environmentContext, _logger);
            BackgroundEnvironment environment = (BackgroundEnvironment) environmentInformation.get("environment");
         
            // Authorize workflow actions from this background environment
            Request request = (Request) environment.getObjectModel().get(ObjectModelHelper.REQUEST_OBJECT);
            AuthenticateAction.setUserIdentityInSession(request, _user, new UserDAO.ImpersonateCredentialProvider(), true);

            // Don't check children twice during this request
            request.setAttribute(__ALREADY_EXPLORING, true);
            
            try
            {
                // Call the workflow
                
                Map<String, Object> inputs = new HashMap<>();
                inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, Map.of(EditContentFunction.QUIT, true));
                // Do action even if the user don't have rights to modify the course
                inputs.put(CheckRightsCondition.FORCE, true);

                Course course = _resolver.resolveById(_courseId);
                _contentWorkflowHelper.doAction(course, 2, inputs);
            }
            catch (AmetysRepositoryException e)
            {
                _logger.warn("An error occurs while resolving the course " + _courseId, e);
            }
            catch (WorkflowException | InvalidActionException e)
            {
                _logger.warn("An error occurs while updating the course " + _courseId, e);
            }
            catch (Exception e)
            {
                _logger.error("Error occurs during threading process", e);
            }
            finally
            {
                BackgroundEngineHelper.leaveEngineEnvironment(environmentInformation);
            }
        }
    }
    
    /**
     * Thread factory for async workflow calls for regime modifications.
     * Set the thread name format and marks the thread as daemon.
     */
    static class AsyncWorkflowCallThreadFactory implements ThreadFactory
    {
        private static ThreadFactory _defaultThreadFactory;
        private static String _nameFormat;
        private static AtomicLong _count;
        
        public AsyncWorkflowCallThreadFactory()
        {
            _defaultThreadFactory = Executors.defaultThreadFactory();
            _nameFormat = "ametys-async-workflow-call-regime-%d";
            _count = new AtomicLong(0);
        }
        
        public Thread newThread(Runnable r)
        {
            Thread thread = _defaultThreadFactory.newThread(r);
            thread.setName(String.format(_nameFormat, _count.getAndIncrement()));
            // make the threads low priority daemon to avoid slowing user thread
            thread.setDaemon(true);
            
            return thread;
        }
    }
}
