/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.helper;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.bouncycastle.util.Arrays;

import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.search.cocoon.ContentGridComponent;
import org.ametys.cms.search.content.ContentValuesExtractorFactory;
import org.ametys.cms.search.content.ContentValuesExtractorFactory.SearchModelContentValuesExtractor;
import org.ametys.cms.search.ui.model.SearchUIModel;
import org.ametys.cms.search.ui.model.SearchUIModelExtensionPoint;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.ui.Callable;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.runtime.model.ViewItemAccessor;

/**
 * The tree helper for grid displaying a view
 */
public class ContentsWithViewTreeGridHelper extends ODFContentsTreeHelper implements Contextualizable
{
    /** The id of the request attribute where required content views are stored */
    protected static final String REQUEST_ATTRIBUTE_SEARCHMODELNAME = ContentsWithViewTreeGridHelper.class.getName() + "$ContentViews";
    
    private Context _context;

    private SearchUIModelExtensionPoint _searchUIModelExtensionPoint;
    private ContentValuesExtractorFactory _contentValuesExtractorFactory;
    private ContentGridComponent _contentGridComponent;
    private ContentWorkflowHelper _contentWorkflowHelper;

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _searchUIModelExtensionPoint = (SearchUIModelExtensionPoint) smanager.lookup(SearchUIModelExtensionPoint.ROLE);
        _contentValuesExtractorFactory = (ContentValuesExtractorFactory) smanager.lookup(ContentValuesExtractorFactory.ROLE);
        _contentGridComponent = (ContentGridComponent) smanager.lookup(ContentGridComponent.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) smanager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    /**
     * Get the root node informations
     * @param contentId The content
     * @param treeId the tree configuration
     * @param searchModel The search model involved
     * @return The informations
     */
    @Callable
    public Map<String, Object> getRootNodeInformations(String contentId, String treeId, String searchModel)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(REQUEST_ATTRIBUTE_SEARCHMODELNAME, searchModel);
        
        return super.getRootNodeInformations(contentId, treeId);
    }
    
    /**
     * Get the node informations
     * @param contentId The content
     * @param searchModel The search model involved
     * @return The informations
     */
    @Callable
    public Map<String, Object> getNodeInformations(String contentId, String searchModel)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(REQUEST_ATTRIBUTE_SEARCHMODELNAME, searchModel);
        
        return getNodeInformations(contentId);
    }
    
    /**
     * Get the children contents according the tree configuration
     * @param contentId the parent content
     * @param treeId the tree configuration
     * @param searchModelName The name of the search model to use to send data
     * @return the children content
     */
    @Callable
    public Map<String, Object> getChildrenContent(String contentId, String treeId, String searchModelName)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(REQUEST_ATTRIBUTE_SEARCHMODELNAME, searchModelName);
        
        return super.getChildrenContent(contentId, treeId);
    }
    
    
    @Override
    protected Map<String, Object> content2Json(Content content)
    {
        Map<String, Object> content2Json = super.content2Json(content);
        
        Request request = ContextHelper.getRequest(_context);
        String searchModelName = (String) request.getAttribute(REQUEST_ATTRIBUTE_SEARCHMODELNAME);
        
        if (searchModelName != null)
        {
            SearchUIModel searchUIModel = _searchUIModelExtensionPoint.getExtension(searchModelName);
            
            Set<String> contentTypesOfModel = searchUIModel.getContentTypes(Map.of());
            
            // Let's see if a view is provided for one of the content types of the content
            boolean jsonContentData = contentTypesOfModel.stream().anyMatch(cType -> _contentTypesHelper.isInstanceOf(content, cType));
            
            if (jsonContentData)
            {
                SearchModelContentValuesExtractor valuesExtractor = _contentValuesExtractorFactory.create(searchUIModel);
                content2Json.put("data", _contentGridComponent.getContentData(content, valuesExtractor, null, Map.of()).get("properties")); // This method returns the repeaters recursively while a simple content.dataToJSON would not
                
                if (Arrays.contains(_contentWorkflowHelper.getAvailableActions((WorkflowAwareContent) content), 2))
                {
                    content2Json.put("notEditableDataIndex", _canEdit(searchUIModel.getResultItems(Map.of()), content));
                }
                else
                {
                    content2Json.put("notEditableData", true);
                }
            }
            else
            {
                // Non supported contents are not editable in the tree grid
                content2Json.put("notEditableData", true);
            }
        }
        else
        {
            // Rootnode contents are not editable in the tree grid (since we do not transfer the search model... yet)
            content2Json.put("notEditableData", true);
        }
        
        return content2Json;
    }
    
    /**
     * List the path that cannot be edited. Not following the content attribute children (since it could imply some items modifiable and others not in a repeater)
     * @param viewItemAcessor The view with definition
     * @param content The content to study
     * @return The list of unmodifiable definition path
     */
    @SuppressWarnings("unchecked")
    protected List<String> _canEdit(ViewItemAccessor viewItemAcessor, Content content)
    {
        List<String> results = new ArrayList<>();
        
        org.ametys.plugins.repository.model.ViewHelper.visitView(viewItemAcessor,
            (element, definition) -> {
                // attribute
                if (definition instanceof RestrictedModelItem restrictedModelItem && !restrictedModelItem.canWrite(content))
                {
                    results.add(definition.getPath());
                }
                
                // content attribute children not handled
            },
            (group, definition) -> {
                // composite
                results.addAll(_canEdit(group, content));
            },
            (group, definition) -> {
                // repeater
                if (definition instanceof RestrictedModelItem restrictedModelItem && !restrictedModelItem.canWrite(content))
                {
                    results.add(definition.getPath());
                }
                else
                {
                    results.addAll(_canEdit(group, content));
                }
            },
            group -> {
                // group
                results.addAll(_canEdit(group, content));
            }
        );
        
        return results;
    }
}
