/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * Class to generate course extract as DOC.
 */
public class MaquetteExtract extends AbstractExtract
{
    @Override
    protected String getType(Map<String, String> reportParameters)
    {
        return "maquette";
    }
    
    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_DOC, OUTPUT_FORMAT_XLS);
    }

    @Override
    public void saxProgramItem(ContentHandler handler, String programItemId, Map<String, String> reportParameters)
    {
        // Check the type of the program item (currently, by default, only programs are accepted)
        ProgramItem programItem = _resolver.resolveById(programItemId);
        if (!(programItem instanceof Program program))
        {
            throw new UnsupportedOperationException("The report '" + getType() + "' can be launch only on programs.");
        }
        
        try
        {
            handler.startDocument();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType());
            XMLUtils.startElement(handler, "report", attrs);

            _reportHelper.saxNaturesEnseignement(handler, getLogger());
            
            AttributesImpl attr = new AttributesImpl();

            attr.addCDATAAttribute("formation", program.getTitle());
            attr.addCDATAAttribute("COD_DIP", _getCodeDIP(program));
            attr.addCDATAAttribute("COD_VDI", _getCodeVRSVDI(program));

            XMLUtils.startElement(handler, "program", attr);

            _saxProgramTree(handler, _reportHelper.getCoursesFromContent(program));

            XMLUtils.endElement(handler, "program");
            
            XMLUtils.endElement(handler, "report");
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'Maquette' extract for program '{}' ({})", program.getTitle(), program.getCode(), e);
        }
    }
    
    @SuppressWarnings("unchecked")
    private void _saxProgramTree(ContentHandler handler, Map<ProgramItem, Object> programTree) throws SAXException
    {
        Map<ProgramItem, Object> sortedProgramTree = new LinkedHashMap<>();
        
        // Order tree elements (course list last, ordered by their type)
        programTree.entrySet()
                .stream()
                .sorted(Map.Entry.<ProgramItem, Object>comparingByKey(new ProgramItemComparator()))
                .forEach(e -> sortedProgramTree.put(e.getKey(), e.getValue()));
        
        for (Entry<ProgramItem, Object> node : sortedProgramTree.entrySet())
        {
            ProgramItem programItem = node.getKey();
            Map<ProgramItem, Object> subTree = (Map<ProgramItem, Object>) node.getValue();
            
            if (programItem instanceof SubProgram subProgram)
            {
                _saxSubProgram(handler, subProgram, subTree);
            }
            else if (programItem instanceof Container container)
            {
                _saxContainer(handler, container, subTree);
            }
            else if (programItem instanceof Course course)
            {
                _saxCourse(handler, course, subTree);
            }
            else if (programItem instanceof CourseList courseList)
            {
                _saxCourseList(handler, courseList, subTree);
            }
        }
    }
    
    private void _saxSubProgram(ContentHandler handler, SubProgram subprogram, Map<ProgramItem, Object> programTree) throws SAXException
    {
        AttributesImpl attr = new AttributesImpl();

        attr.addCDATAAttribute("parcours", subprogram.getTitle());

        XMLUtils.startElement(handler, "subprogram", attr);

        _saxProgramTree(handler, programTree);

        XMLUtils.endElement(handler, "subprogram");
    }

    private void _saxContainer(ContentHandler handler, Container container, Map<ProgramItem, Object> programTree) throws SAXException
    {
        AttributesImpl attr = new AttributesImpl();

        attr.addCDATAAttribute("id", container.getId());
        attr.addCDATAAttribute("libelle", container.getTitle());

        String containerNature = _refTableHelper.getItemCode(container.getNature());
        attr.addCDATAAttribute("type", containerNature);
        if ("annee".equals(containerNature))
        {
            attr.addCDATAAttribute("COD_ETP", container.getValue("etpCode", false, StringUtils.EMPTY));
            attr.addCDATAAttribute("COD_VRS_ETP", container.getValue("vrsEtpCode", false, StringUtils.EMPTY));
        }

        long codeAnu = container.getValue("CodeAnu", false, 0L);
        if (codeAnu > 0)
        {
            attr.addCDATAAttribute("HeaderCodeAnu", "Maquette " + String.valueOf(codeAnu) + "-" + String.valueOf(codeAnu + 1));
        }

        XMLUtils.startElement(handler, "container", attr);

        _saxProgramTree(handler, programTree);

        XMLUtils.endElement(handler, "container");
    }

    private void _saxCourse(ContentHandler handler, Course course, Map<ProgramItem, Object> programTree) throws SAXException
    {
        AttributesImpl attr = new AttributesImpl();
        
        attr.addCDATAAttribute("CodeAmetysELP", course.getCode());
        attr.addCDATAAttribute("CodeApogee", course.getValue("elpCode", false, StringUtils.EMPTY));
        attr.addCDATAAttribute("natureElement", _refTableHelper.getItemCode(course.getCourseType()));
        attr.addCDATAAttribute("ects", String.valueOf(course.getEcts()));
        
        Container containerEtape = Optional.ofNullable((ContentValue) course.getValue("etapePorteuse"))
                                           .flatMap(contentValue -> _getEtapePorteuseIfExists(contentValue, course))
                                           .map(Container.class::cast)
                                           .orElse(null);
        
        if (containerEtape != null)
        {
            attr.addCDATAAttribute("etapePorteuse", containerEtape.getTitle());
            attr.addCDATAAttribute("idEtapePorteuse", containerEtape.getId());
            if (containerEtape.hasValue("etpCode"))
            {
                attr.addCDATAAttribute("codeEtapePorteuse", containerEtape.getValue("etpCode") + (containerEtape.hasValue("vrsEtpCode") ? "-" + containerEtape.getValue("vrsEtpCode") : StringUtils.EMPTY));
            }
        }
        
        if (course.hasValue("CodeAnu"))
        {
            long codeAnu = course.getValue("CodeAnu");
            attr.addCDATAAttribute("HeaderCodeAnu", "Maquette " + String.valueOf(codeAnu) + "-" + String.valueOf(codeAnu + 1));
        }

        XMLUtils.startElement(handler, "course", attr);

        String shortLabel = course.getValue("shortLabel");
        if (StringUtils.isNotBlank(shortLabel))
        {
            XMLUtils.createElement(handler, "shortLabel", shortLabel);
        }
        XMLUtils.createElement(handler, "libelle", course.getTitle());

        long courseListsSize = course.getParentCourseLists().size();
        
        // Partagé
        XMLUtils.createElement(handler, "partage", courseListsSize > 1 ? "X" : "");

        // Nb occurrences
        XMLUtils.createElement(handler, "occurrences", _reportHelper.formatNumberToSax(courseListsSize));
        
        // Heures d'enseignement
        for (CoursePart coursePart : course.getCourseParts())
        {
            attr = new AttributesImpl();
            attr.addCDATAAttribute("nature", coursePart.getNature());
            XMLUtils.createElement(handler, "volumeHoraire", attr, String.valueOf(coursePart.getNumberOfHours()));
        }
        
        _saxProgramTree(handler, programTree);
        
        XMLUtils.endElement(handler, "course");

    }
    
    private Optional<ModifiableContent> _getEtapePorteuseIfExists(ContentValue etapePorteuse, Course course)
    {
        try
        {
            return Optional.ofNullable(etapePorteuse.getContent());
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().error("Course {} {}", course.getId(), e.getMessage(), e);
            return Optional.empty();
        }
    }

    private void _saxCourseList(ContentHandler handler, CourseList courseList, Map<ProgramItem, Object> programTree) throws SAXException
    {
        AttributesImpl attr = new AttributesImpl();
        ChoiceType typeList = courseList.getType();
        if (typeList.name().equals(ChoiceType.CHOICE.toString()) && courseList.getMinNumberOfCourses() > 0)
        {
            attr.addCDATAAttribute("choice", String.valueOf(courseList.getMinNumberOfCourses()));
        }
        else if (typeList.name().equals(ChoiceType.OPTIONAL.toString()))
        {
            attr.addCDATAAttribute("optional", "true");
        }
        XMLUtils.startElement(handler, "courselist", attr);

        _saxProgramTree(handler, programTree);

        XMLUtils.endElement(handler, "courselist");
    }
    
    private static final class ProgramItemComparator implements Comparator<ProgramItem>
    {
        public int compare(ProgramItem p1, ProgramItem p2)
        {
            if (p1 instanceof CourseList && !(p2 instanceof CourseList))
            {
                // Order courselist after subprogram or container
                return 1;
            }
            else if (!(p1 instanceof CourseList) && p2 instanceof CourseList)
            {
                // Order courselist after subprogram or container
                return -1;
            }
            else if (p1 instanceof CourseList cl1 && p2 instanceof CourseList cl2)
            {
                // Order courselist according to their type (mandatory > choice > optional)
                ChoiceType t1 = cl1.getType();
                ChoiceType t2 = cl2.getType();
                
                if (t1.equals(t2))
                {
                    return 0;
                }
                else if (t1.equals(ChoiceType.MANDATORY) || !t2.equals(ChoiceType.MANDATORY) && t1.equals(ChoiceType.CHOICE))
                {
                    return -1;
                }
                else
                {
                    return 1;
                }
            }
            else
            {
                // No order between other program items (subprograms or semesters)
                return 0;
            }
        }
    }
    
    /**
     * Get code VRSVDI
     * @param content the content
     * @return the codeVRSVDI if it's set, otherwise the second part of the content code
     */
    private String _getCodeVRSVDI(ModifiableDefaultContent content)
    {
        String defaultCode = StringUtils.substringAfter(((ProgramItem) content).getCode(), "-");
        return content.getValue("codeVRSVDI", false, defaultCode);
    }
    
    /**
     * Get code DIP
     * @param content the content
     * @return the codeDIP if it's set, otherwise the first part of the content code
     */
    private String _getCodeDIP(ModifiableDefaultContent content)
    {
        String defaultCode = StringUtils.substringBefore(((ProgramItem) content).getCode(), "-");
        return content.getValue("codeDIP", false, defaultCode);
    }
}
