/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl.mcc;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.ametys.cms.content.compare.ContentComparatorResult;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.data.EducationalPath;

/**
 * Structure representing a tree of {@link ProgramItem} to be used during the
 * processing of the MCC reports.
 */
public class MCCProgramItemTree
{
    private MCCProgramItemTree _parent;
    private ProgramItem _current;
    private List<MCCProgramItemTree> _children = new ArrayList<>();
    private ContentComparatorResult _change;
    private EducationalPath _path;

    private MCCProgramItemTree(MCCProgramItemTree parent, ProgramItem current, ContentComparatorResult change)
    {
        _parent = parent;
        _current = current;
        _change = change;
    }
    
    private MCCProgramItemTree(MCCProgramItemTree parent, ProgramItem current)
    {
        this(parent, current, null);
    }
    
    /**
     * The constructor
     * @param current The {@link ProgramItem} the tree is build on
     */
    public MCCProgramItemTree(ProgramItem current)
    {
        this(null, current);
    }
    
    /**
     * The constructor
     * @param current The {@link ProgramItem} the tree is build on
     * @param change changes for this item (if applicable)
     */
    public MCCProgramItemTree(ProgramItem current, ContentComparatorResult change)
    {
        this(null, current, change);
    }

    /**
     * Declare a child for the current object.
     * @param child child for this node
     * @return The corresponding {@link MCCProgramItemTree} for the child
     */
    public MCCProgramItemTree addChild(ProgramItem child)
    {
        return addChild(child, null);
    }
    
    /**
     * Declare a child for the current object.
     * 
     * @param child child for this node
     * @param change changes for this child (if applicable)
     * @return The corresponding {@link MCCProgramItemTree} for the child
     */
    public MCCProgramItemTree addChild(ProgramItem child, ContentComparatorResult change)
    {
        MCCProgramItemTree childTree = _createChild(child, change);
        _addChild(childTree);
        return childTree;
    }

    /**
     * Add a new child tree
     * @param childTree The child tree to add
     */
    protected void _addChild(MCCProgramItemTree childTree)
    {
        _children.add(childTree);
    }
    
    /**
     * Create a child for this node, without adding it to the tree (usefull for tests before adding, addChild to add)
     * @param child The child {@link ProgramItem}
     * @param change Changes for this child (if applicable)
     * @return A child tree
     */
    private MCCProgramItemTree _createChild(ProgramItem child, ContentComparatorResult change)
    {
        return new MCCProgramItemTree(this, child, change);
    }

    /**
     * Get the parent ametys object
     * @return {@link MCCProgramItemTree} or null
     */
    public MCCProgramItemTree getParent()
    {
        return _parent;
    }

    /**
     * Get the current ametys object (root level of the tree).
     * @return {@link ProgramItem}
     */
    public ProgramItem getCurrent()
    {
        return _current;
    }

    /**
     * Get the current ametys object (root level of the tree).
     * @return {@link List} of {@link MCCProgramItemTree}
     */
    public List<MCCProgramItemTree> getChildren()
    {
        return _children;
    }

    /**
     * Get the change informations for this node
     * @return {@link ContentComparatorResult} or null if not applicable
     */
    public ContentComparatorResult getChange()
    {
        return _change;
    }
    
    /**
     * Get the path in the tree.
     * orgunits are excluded.
     * @return the path as a {@link String}
     */
    public EducationalPath getPath()
    {
        // Compute the path if it does not exist
        // Only manage program items
        if (_path == null)
        {
            return Optional.ofNullable(_parent)
                .map(MCCProgramItemTree::getPath)
                .map(parentPath -> EducationalPath.of(parentPath, _current))
                .orElseGet(() -> EducationalPath.of(_current));
        }
        
        return _path;
    }
}

