/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.pipeline;

import java.io.IOException;
import java.time.LocalDate;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.serialization.ZipArchiveSerializer;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.odfpilotage.report.AbstractPilotageReport;
import org.ametys.plugins.odfpilotage.report.PilotageReport;
import org.ametys.plugins.odfpilotage.report.PilotageReport.PilotageReportContent;
import org.ametys.plugins.odfpilotage.report.PilotageReport.PilotageReportTarget;

/**
 * Generator to prepare a ZIP for pilotage reports.
 */
public class PilotageReportZipGenerator extends ServiceableGenerator
{
    private JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        PilotageReport pilotageReport = (PilotageReport) request.getAttribute("report");
        PilotageReportTarget target = (PilotageReportTarget) request.getAttribute("reportTarget");
        @SuppressWarnings("unchecked")
        Map<String, String> reportParameters = (Map<String, String>) request.getAttribute("reportParameters");
        
        contentHandler.startDocument();
        contentHandler.startPrefixMapping("zip", ZipArchiveSerializer.ZIP_NAMESPACE);
        XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");
        
        PilotageReportContent content = pilotageReport.getReportContent(target, reportParameters);

        // Set request attributes to launch each report file
        request.setAttribute("reportPluginName", pilotageReport.getPluginName());
        
        Map<String, String> files = content.files();
        if (files.isEmpty())
        {
            // Set request attribute because there is no report to generate
            request.setAttribute("reportStatus", "NO_FILE");
        }
        else
        {
            _createZipEntries(files);
            _addManifestEntry(pilotageReport, target, reportParameters);
            
            // Set request attributes to build the final ZIP
            request.setAttribute("reportZipName", content.zipName());
        }
        
        XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");
        contentHandler.endPrefixMapping("zip");
        contentHandler.endDocument();
    }
    
    private void _addManifestEntry(PilotageReport report, PilotageReportTarget target, Map<String, String> reportParameters) throws SAXException
    {
        Map<String, Object> manifestData = new HashMap<>();
        manifestData.put("type", report.getId());
        manifestData.put("date", LocalDate.now());
        manifestData.put("target", target.name().toLowerCase());
        manifestData.putAll(reportParameters);
        
        AttributesImpl zipAttrs = new AttributesImpl();
        zipAttrs.addCDATAAttribute("name", AbstractPilotageReport.MANIFEST_FILENAME);
        zipAttrs.addCDATAAttribute("serializer", "text");
        XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry", zipAttrs);
        XMLUtils.createElement(contentHandler, "text", _jsonUtils.convertObjectToJson(manifestData));
        XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry");
    }
    
    private void _createZipEntries(Map<String, String> filesList) throws SAXException
    {
        for (String filename : filesList.keySet())
        {
            AttributesImpl zipAttrs = new AttributesImpl();
            zipAttrs.addCDATAAttribute("name", filename);
            zipAttrs.addCDATAAttribute("src", filesList.get(filename));
            XMLUtils.createElementNS(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry", zipAttrs);
        }
    }
}
