/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.workflow;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeaterEntry;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Workflow function to create unique code on additional thematics and rules of a container.
 */
public class ContainerThematicCodeEditionFunction extends AbstractCodeEditionFunction
{
    /** The rules manager */
    protected PilotageHelper _pilotageHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _pilotageHelper = (PilotageHelper) smanager.lookup(PilotageHelper.ROLE);
    }
    
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        WorkflowAwareContent content = getContent(transientVars);
        if (_pilotageHelper.isContainerOfTypeYear(content))
        {
            ModifiableIndexableRepeater repeater = ((ModifiableContent) content).getRepeater(RulesManager.CONTAINER_THEMATICS);
            if (repeater != null)
            {
                String prefix = RulesManager.ADDITIONAL_THEMATICS_PREFIX;
                Long codeNumber = content.getInternalDataHolder().getValue(prefix, 0L);
                for (ModifiableIndexableRepeaterEntry entry : repeater.getEntries())
                {
                    if (!entry.hasValue(RulesManager.THEMATIC_CODE))
                    {
                        codeNumber++;
                        entry.setValue(RulesManager.THEMATIC_CODE, prefix + codeNumber);
                    }
                    
                    _setRulesCode((ModifiableContent) content, entry, RulesManager.CONTAINER_RULES);
                }
                content.getInternalDataHolder().setValue(prefix, codeNumber);
                content.saveChanges();
            }
        }
    }
    
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CONTAINER_RULE_CODE_EDITION_FUNCTION_LABEL");
    }
}
