/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/**
 * Show the values of a repeater in a dialog, and allow the inline modification of its entries
 */
Ext.define('Ametys.plugins.odfpilotage.form.widget.DoubleMCCSearchGridRepeater', {
    singleton: true,
    
    showRepeaterDialog: function()
    {
        return Ametys.plugins.cms.search.SearchGridRepeaterDialog.showRepeaterDialog.apply(this, arguments);
    },
    
    _openDialog: function (title, parentRecord, contentId, subcolumns, repeaterCfg, dataIndex, metadataPath, contentGridId, callback)
    {
        let items = [];
        let grids = [];
        
        for (let i = 0; i < subcolumns.length; i++)
        {
            let items1 = Ametys.plugins.cms.search.SearchGridRepeaterDialog._createItems(title, parentRecord, contentId, subcolumns[i], repeaterCfg[i], dataIndex[i], metadataPath[i], contentGridId);
            grids.push(items1[0]);
            items.push({
                xtype: 'panel', 
                flex: 1,
                layout: 'fit',
                title: title[i],
                items: items1
            });
        }
        
        items.splice(1, 0, {
            xtype: 'container',
            layout: 'hbox',
            style: {
                marginTop: '20px'
            },
            items: [
                { xtype: 'tbspacer', flex: 1 },
                {
                    xtype: 'button',
                    text: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_TOOL_MCCCOURSE_COPY_LABEL}}",
                    handler: Ext.bind(this._copySessions, this, [dataIndex[dataIndex.length - 1]], 1),
                    tooltip: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_SESSIONS_COPY_TOOLTIP}}"
                },
                { xtype: 'tbspacer', flex: 1 }
            ]
        });
        
        var isGrid1ReadOnly = parentRecord.data['notEditableData'] === true || parentRecord.data['notEditableDataIndex'] && Ext.Array.contains(parentRecord.data['notEditableDataIndex'], metadataPath[0]);
        var isGrid2ReadOnly = parentRecord.data['notEditableData'] === true || parentRecord.data['notEditableDataIndex'] && Ext.Array.contains(parentRecord.data['notEditableDataIndex'], metadataPath[1]);
        var dialog = Ametys.plugins.cms.search.SearchGridRepeaterDialog._showDialog(
            "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_DIALOG_MCCCOURSE_LABEL}} " + parentRecord.get('title'), 
            items, 
            {type: 'vbox', align: 'stretch'},
            function () {
                Ametys.plugins.odfpilotage.form.widget.DoubleMCCSearchGridRepeater._ok(dialog, parentRecord, grids, dataIndex, contentId, callback);
            },
            callback,
            [
                {
                    text :"{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_SESSIONS_BUTTON_HANDLE}}",
                    handler: function() {
                        Ametys.plugins.odf.pilotage.helper.DoubleMCCTemplatesHelper.handleMCCSessionsTemplates(grids, parentRecord.get("mccRegime"), !(isGrid1ReadOnly || repeaterCfg[0].disabled), !(isGrid2ReadOnly || repeaterCfg[1].disabled));
                    },
                    tooltip: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_SESSIONS_HANDLE_TOOLTIP}}",
                    scope: this
                },
                {
                    text :"{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_SESSIONS_BUTTON_CREATE}}",
                    handler: function() {
                        Ametys.plugins.odf.pilotage.helper.DoubleMCCTemplatesHelper.createTemplate(grids);
                    },
                    tooltip: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_SESSIONS_CREATE_TOOLTIP}}",
                    itemId: 'create-user-session',
                    scope: this
                },
                { xtype: 'tbspacer', flex: 1 }
            ]
        );
        
        this._initCreateTemplateButton(dialog, grids);
        
        dialog.setReadOnly = function() {
            let copyButton = dialog.items.get(1); 
            copyButton.disable();
            for (let i = 0; i < dialog.items.getCount(); i+=2)
            {
                dialog.items.get(i).items.get(0).getStore().fireEvent('datachanged');
            }
            
            if (dialog.items.get(2).items.get(0).getStore().getCount() == 0)
            {
                copyButton.hide();
                dialog.items.get(2).hide();
                dialog.setHeight(500);
            }
        }
        
        if (isGrid2ReadOnly)
        {
            dialog.setReadOnly();
        }
        else if (repeaterCfg[1].disabled)
        {
            let copyButton = dialog.items.get(1); 
            copyButton.disable();
        }
        
        return dialog;
    },
    
    _initCreateTemplateButton: function(dialog, grids)
    {
        let storeGrid1 = grids[0].getStore();
        let storeGrid2 = grids[1].getStore();
        let createTemplateButton = dialog.down("#create-user-session");
        
        if (storeGrid1.getCount() == 0 && storeGrid2.getCount() == 0)
        {
            createTemplateButton.disable();
        }
        else
        {
            createTemplateButton.enable();
        }
        
        storeGrid1.on("datachanged", function() {
            if (storeGrid1.getCount() == 0 && storeGrid2.getCount() == 0)
            {
                createTemplateButton.disable();
            }
            else
            {
                createTemplateButton.enable();
            }
        })
        
        storeGrid2.on("datachanged", function() {
            if (storeGrid1.getCount() == 0 && storeGrid2.getCount() == 0)
            {
                createTemplateButton.disable();
            }
            else
            {
                createTemplateButton.enable();
            }
        })
    },
    
    _copySessions: function(button, config)
    {
        let sourceStore = button.ownerCt.prev().items.get(0).store;
        let targetStore = button.ownerCt.next().items.get(0).store;
        
        let dataThatRequiresAServerConversion = [];
        let toReplace = {};
        let newRecords = {};
        
        for (let i=0; i < sourceStore.getCount(); i++)
        {
            let record = sourceStore.getAt(i);
            if (record.get(config.authorizeCopy) != true)
            {
                let newRecord = targetStore.getModel().create();
                
                let data = {};
                
                for (let d in record.data)
                {
                    if (d != 'id' && d != config.authorizeCopy)
                    {
                        if (!(config.doNotCopy || []).includes(d))
                        {
                            // Direct copy
                            data[d] = record.data[d];
                        }
                        else if (typeof record.data[d] == 'string' && record.data[d])
                        {
                            toReplace[newRecord.getId()] = toReplace[newRecord.getId()] || [];
                            toReplace[newRecord.getId()].push(d);
                             
                            data[d] = record.data[d];
                            if (!dataThatRequiresAServerConversion.includes(record.data[d]))
                            {
                                // Require a server conversion of the value
                                dataThatRequiresAServerConversion.push(record.data[d]);
                            }
                        }
                    }
                }
                
                newRecords[newRecord.getId()] = {
                    record: newRecord,
                    data: data
                }
            }
        }
        
        if (dataThatRequiresAServerConversion.length == 0)
        {
            this._copySessionsStep2(null, {store: targetStore, newRecords: newRecords, toReplace: toReplace});
        }
        else
        {
            Ametys.data.ServerComm.callMethod({
                role: "org.ametys.plugins.odfpilotage.helper.PilotageHelper",
                methodName: "getCompatibleModalityInSession2",
                parameters: [dataThatRequiresAServerConversion],
                callback: {
                    handler: this._copySessionsStep2,
                    scope: this,
                    arguments: {
                        store: targetStore,
                        newRecords: newRecords,
                        toReplace: toReplace
                    }
                },
                errorMessage: true
            });            
        }
    },
    
    _copySessionsStep2: function(serverData, args)
    {
        let store = args.store;
        let newRecords = args.newRecords;
        let toReplace = args.toReplace;
        
        for (let newRecordId in newRecords)
        {
            let newRecord = newRecords[newRecordId].record;
            let data = newRecords[newRecordId].data;
            
            store.add(newRecord);
            newRecord.set(data); // setting data after, to be not commited
            
            if (serverData && toReplace[newRecord.getId()])
            {
                for (let k of toReplace[newRecord.getId()])
                {
                    newRecord.set(k, serverData[newRecord.get(k)] || null);
                }
            }
            
            Ametys.plugins.cms.search.SearchGridRepeaterDialog._markAllAsModified(null, newRecord); // to ensure, empty data have the red flag
        }
    },
    
    _ok: function (dialog, parentRecord, grids, dataIndex, contentId, callback)
    {
        let repeaterValuesAll = {};
        
        for (let i=0; i < grids.length; i++)
        {
            var entriesValues = Ametys.plugins.cms.search.SearchGridRepeaterDialog._gridToValues(grids[i]);
            if (entriesValues == null)
            {
                Ametys.log.ErrorDialog.display({
                    title: "{{i18n plugin.cms:UITOOL_CONTENTEDITIONGRID_REPEATER_VALUE_ERROR_TITLE}}",
                    text: "{{i18n plugin.cms:UITOOL_CONTENTEDITIONGRID_REPEATER_VALUE_ERROR_DESC}}" + "<br/>" + grids[i].ownerCt.getTitle(),
                    details: "The repeater n°" + i + " has some invalid values",
                    category: this.self.getName()
                });
                
                return;
            }
            
            var repeaterValues = parentRecord.getData()[dataIndex[i] + "_repeater"];
            repeaterValues.entries = [];
            for (var index = 0; index < entriesValues.length; index++)
            {
                repeaterValues.entries[index] = {
                    position: index + 1,
                    values: entriesValues[index]
                };
            }
            
            repeaterValuesAll[dataIndex[i]] = repeaterValues;
        }

        if (callback)
        {
            callback(repeaterValuesAll);
            dialog.doNotCallCallback = true;
        }
        
        Ametys.plugins.cms.search.SearchGridRepeaterDialog._closeForced = true;
        dialog.close();
    },
});
