/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.clientsideelement;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.SchedulerException;

import org.ametys.cms.clientsideelement.SmartContentClientSideElement;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.schedule.Runnable;
import org.ametys.core.schedule.Runnable.FireProcess;
import org.ametys.core.schedule.Runnable.MisfirePolicy;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.SchedulableExtensionPoint;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.odf.program.Container;
import org.ametys.plugins.core.impl.schedule.DefaultRunnable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowException;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowStep;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.schedulable.MCCValidatedPDFSchedulable;
import org.ametys.plugins.repository.lock.LockHelper;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Client side element for MCC workflow buttons on a {@link Container}
 */
public class MCCWorkflowClientSideElement extends SmartContentClientSideElement
{
    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;

    /** The pilotage helper */
    protected PilotageHelper _pilotageHelper;

    /** The scheduler */
    protected Scheduler _scheduler;
    
    /** The scheduler extention point */
    protected SchedulableExtensionPoint _schedulerEP;
    
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _mccWorkflowHelper = (MCCWorkflowHelper) manager.lookup(MCCWorkflowHelper.ROLE);
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
        _scheduler = (Scheduler) manager.lookup(Scheduler.ROLE);
        _schedulerEP = (SchedulableExtensionPoint) manager.lookup(SchedulableExtensionPoint.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        List<Script> scripts = super.getScripts(ignoreRights, contextParameters);
        
        if (!scripts.isEmpty())
        {
            MCCWorkflowStep mccStatus = MCCWorkflowStep.valueOf((String) scripts.get(0).getParameters().get("mcc-status"));
            if (MCCWorkflowStep.RULES_VALIDATED.equals(mccStatus))
            {
                if (!RulesManager.isRulesEnabled())
                {
                    return List.of();
                }
            }
        }
        return scripts;
    }
    
    /**
     * Get informations on contents' state
     * @param containerIds the ids of container
     * @param mccStatus the MCC status
     * @return informations on contents' state
     */
    @Callable
    public Map<String, Object> getStatus(List<String> containerIds, String mccStatus)
    {
        Map<String, Object> results = super.getStatus(containerIds);
        
        results.put("active-contents", new ArrayList<>());
        results.put("invalidmccstatus-contents", new ArrayList<>());
        results.put("noyear-contents", new ArrayList<>());
        
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> activeContents = (List<Map<String, Object>>) results.get("active-contents");
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> invalidContents = (List<Map<String, Object>>) results.get("invalidmccstatus-contents");
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> noYearContents = (List<Map<String, Object>>) results.get("noyear-contents");
        
        for (String containerId : containerIds)
        {
            Container container = _resolver.resolveById(containerId);
            
            _getStatus(container, mccStatus, activeContents, noYearContents, invalidContents);
        }
        
        List<Object> invalidContentIds = invalidContents.stream()
            .map(p -> p.get("id"))
            .toList();
        
        List<Object> noYearContentIds = noYearContents.stream()
                .map(p -> p.get("id"))
                .toList();
        
        // Remove content with invalid MCC status from all right contents
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> allrightContents = (List<Map<String, Object>>) results.get("allright-contents");
        results.put("allright-contents", allrightContents.stream().filter(p -> !invalidContentIds.contains(p.get("id")) && !noYearContentIds.contains(p.get("id"))).toList());
        
        return results;
    }
    
    /**
     * Get status for a container
     * @param container the container
     * @param mccStatus the MCC status
     * @param activeContents The list of active contents
     * @param noYearContents The list of no-year containers
     * @param invalidMCCStatusContents the list of container with invalid MCC status for the current action
     */
    protected void _getStatus(Container container, String mccStatus, List<Map<String, Object>> activeContents, List<Map<String, Object>> noYearContents, List<Map<String, Object>> invalidMCCStatusContents)
    {
        Map<String, Object> defaultContentParams = getContentDefaultParameters(container);
        
        if (!_pilotageHelper.isContainerOfTypeYear(container))
        {
            Map<String, Object> noYearContentParams = new HashMap<>(defaultContentParams);
            noYearContentParams.put("description", _getNoYearDescription(container));
            noYearContents.add(noYearContentParams);
            return;
        }
        
        switch (MCCWorkflowStep.valueOf(mccStatus))
        {
            case RULES_VALIDATED:
                if (_mccWorkflowHelper.isRulesValidated(container))
                {
                    Map<String, Object> activeContentParams = new HashMap<>(defaultContentParams);
                    activeContentParams.put("description", _getActiveDescription(container));
                    activeContents.add(activeContentParams);
                }
                
                if (_mccWorkflowHelper.isMCCOrgunitValidated(container) || !_mccWorkflowHelper.canValidateRules(container))
                {
                    Map<String, Object> invalidContentParams = new HashMap<>(defaultContentParams);
                    invalidContentParams.put("description", _getInvalidMCCStatusDescription(container));
                    invalidMCCStatusContents.add(invalidContentParams);
                }
                break;
            case MCC_VALIDATED:
                if (_mccWorkflowHelper.isMCCValidated(container))
                {
                    Map<String, Object> activeContentParams = new HashMap<>(defaultContentParams);
                    activeContentParams.put("description", _getActiveDescription(container));
                    activeContents.add(activeContentParams);
                }
                
                if (_mccWorkflowHelper.isMCCOrgunitValidated(container) || !_mccWorkflowHelper.canValidateMCC(container))
                {
                    Map<String, Object> invalidContentParams = new HashMap<>(defaultContentParams);
                    invalidContentParams.put("description", _getInvalidMCCStatusDescription(container));
                    invalidMCCStatusContents.add(invalidContentParams);
                }
                break;
            case MCC_ORGUNIT_VALIDATED:
                if (_mccWorkflowHelper.isMCCOrgunitValidated(container))
                {
                    Map<String, Object> activeContentParams = new HashMap<>(defaultContentParams);
                    activeContentParams.put("description", _getActiveDescription(container));
                    activeContents.add(activeContentParams);
                }
                
                if (_mccWorkflowHelper.isMCCCFVUValidated(container) || !_mccWorkflowHelper.canValidateOrgUnitMCC(container))
                {
                    Map<String, Object> invalidContentParams = new HashMap<>(defaultContentParams);
                    invalidContentParams.put("description", _getInvalidMCCStatusDescription(container));
                    invalidMCCStatusContents.add(invalidContentParams);
                }
                break;
            case CFVU_MCC_VALIDATED:
                if (_mccWorkflowHelper.isMCCCFVUValidated(container))
                {
                    Map<String, Object> activeContentParams = new HashMap<>(defaultContentParams);
                    activeContentParams.put("description", _getActiveDescription(container));
                    activeContents.add(activeContentParams);
                }
                
                if (!_mccWorkflowHelper.canValidateCFVUMCC(container))
                {
                    Map<String, Object> invalidContentParams = new HashMap<>(defaultContentParams);
                    invalidContentParams.put("description", _getInvalidMCCStatusDescription(container));
                    invalidMCCStatusContents.add(invalidContentParams);
                }
                break;
            default:
                throw new IllegalArgumentException("Unexpected value for MCC status : " + mccStatus);
        }
    }
    
    /**
     * Get i18n description when the MCC workflow status is active
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getActiveDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("active-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
    
    /**
     * Get i18n description when the MCC workflow status does not allow to do action
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getInvalidMCCStatusDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("invalidmccstatus-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
    
    /**
     * Get i18n description when the container is not of type year
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getNoYearDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("noyear-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
    
    /**
     * Validate the rules
     * @param containerIds the container ids
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> validateRules(List<String> containerIds, String validationDate, String comment)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        ZonedDateTime zonedDateTime = DateUtils.parseZonedDateTime(validationDate).withZoneSameInstant(ZoneId.systemDefault());
        LocalDate localDate = zonedDateTime != null ? zonedDateTime.toLocalDate() : LocalDate.now();
        
        return _doMCCAction(containerIds, "ODF_Pilotage_MCC_Rules_Validated_Rights", container -> _mccWorkflowHelper.validateRules(container, localDate, user, comment));
    }
    
    /**
     * Invalidate the rules
     * @param containerIds the container ids
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> invalidateRules(List<String> containerIds)
    {
        return _doMCCAction(containerIds, "ODF_Pilotage_MCC_Rules_Validated_Rights", container -> _mccWorkflowHelper.invalidateRules(container));
    }
    
    /**
     * Validate the rules
     * @param containerIds the container ids
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> validateMCC(List<String> containerIds, String validationDate, String comment)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        ZonedDateTime zonedDateTime = DateUtils.parseZonedDateTime(validationDate).withZoneSameInstant(ZoneId.systemDefault());
        LocalDate localDate = zonedDateTime != null ? zonedDateTime.toLocalDate() : LocalDate.now();
        
        return _doMCCAction(containerIds, "ODF_Pilotage_MCC_Validated_Rights", container -> _mccWorkflowHelper.validateMCC(container, localDate, user, comment));
    }
    
    /**
     * Invalidate the rules
     * @param containerIds the container ids
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> invalidateMCC(List<String> containerIds)
    {
        return _doMCCAction(containerIds, "ODF_Pilotage_MCC_Validated_Rights", container -> _mccWorkflowHelper.invalidateMCC(container));
    }
    
    /**
     * Get minimun date for MCC orgunit validation for given containers
     * @param containerIds the container ids
     * @return the minimun date
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public LocalDate getMinDateForMCCOrgUnitValidation(List<String> containerIds)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate minDate = null;
        for (String containerId : containerIds)
        {
            Container container = _resolver.resolveById(containerId);
            
            if (_rightManager.hasRight(user, "ODF_Pilotage_MCC_Orgunit_Validated_Rights", container) == RightResult.RIGHT_ALLOW)
            {
                LocalDate cMinDate = _mccWorkflowHelper.getMinDateForMCCOrgUnitValidation(container);
                if (minDate == null || cMinDate.isAfter(minDate))
                {
                    minDate = cMinDate;
                }
            }
        }
        
        return minDate;
    }
    
    /**
     * Validate the MCC to orgunit level
     * @param containerIds the container ids
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> validateOrgunitMCC(List<String> containerIds, String validationDate, String comment)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        ZonedDateTime zonedDateTime = DateUtils.parseZonedDateTime(validationDate).withZoneSameInstant(ZoneId.systemDefault());
        LocalDate localDate = zonedDateTime != null ? zonedDateTime.toLocalDate() : LocalDate.now();
        
        return _doMCCAction(containerIds, "ODF_Pilotage_MCC_Orgunit_Validated_Rights", container -> _mccWorkflowHelper.validateOrgunitMCC(container, localDate, user, comment));
    }
    
    /**
     * Invalidate the MCC for orgunit level
     * @param containerIds the container ids
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> invalidateOrgunitMCC(List<String> containerIds)
    {
        return _doMCCAction(containerIds, "ODF_Pilotage_MCC_Orgunit_Validated_Rights", container -> _mccWorkflowHelper.invalidateOrgunitMCC(container));
    }
    
    /**
     * Get minimun date for MCC CFVU validation for given container
     * @param containerIds the container ids
     * @return the minimun date for MCC CFVU validation
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public LocalDate getMinDateForMCCCFVUValidation(List<String> containerIds)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate minDate = null;
        for (String containerId : containerIds)
        {
            Container container = _resolver.resolveById(containerId);
            
            if (_rightManager.hasRight(user, "ODF_Pilotage_CFVU_MCC_Validated_Rights", container) == RightResult.RIGHT_ALLOW)
            {
                LocalDate cMinDate = _mccWorkflowHelper.getMinDateForMCCCFVUValidation(container);
                if (minDate == null || cMinDate.isAfter(minDate))
                {
                    minDate = cMinDate;
                }
            }
        }
        
        return minDate;
        
    }
    
    /**
     * Validate the MCC
     * @param containerIds the container ids
     * @param validationDate the date of validation. Can be empty or null.
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> validateMCCForCVFU(List<String> containerIds, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        ZonedDateTime zonedDateTime = DateUtils.parseZonedDateTime(validationDate).withZoneSameInstant(ZoneId.systemDefault());
        LocalDate localDate = zonedDateTime != null ? zonedDateTime.toLocalDate() : LocalDate.now();
        
        Map<String, Object> results = _doMCCAction(containerIds, "ODF_Pilotage_CFVU_MCC_Validated_Rights", container -> _mccWorkflowHelper.validateMCCForCVFU(container, localDate, user, comment));
        
        try
        {
            @SuppressWarnings("unchecked")
            List<String> successContentIds = (List<String>) results.get("success-contents");
            if (!successContentIds.isEmpty())
            {
                Map<String, Object> parameters = new HashMap<>();
                parameters.put("date", localDate);
                parameters.put("containerIds", successContentIds);
                parameters.put("contextualParameters", _jsonUtils.convertObjectToJson(contextualParameters));
                
                Schedulable mccValidatedPDFSchedulable = _schedulerEP.getExtension(MCCValidatedPDFSchedulable.ID);
                String runnableId = mccValidatedPDFSchedulable.getId() + "$" + UUID.randomUUID();
                Runnable runnable = new DefaultRunnable(runnableId,
                        mccValidatedPDFSchedulable.getLabel(),
                        mccValidatedPDFSchedulable.getDescription(),
                        FireProcess.NOW,
                        null /* cron*/,
                        mccValidatedPDFSchedulable.getId(),
                        true /* removable */,
                        false /* modifiable */,
                        false /* deactivatable */,
                        MisfirePolicy.FIRE_ONCE,
                        true /* isVolatile */,
                        _currentUserProvider.getUser(),
                        parameters
                    );
                    
                _scheduler.scheduleJob(runnable);
                results.put("schedulable-id", runnableId);
                results.put("schedulable-label", mccValidatedPDFSchedulable.getLabel());
            }
        }
        catch (SchedulerException e)
        {
            results.put("error", "schedulable-error");
        }
        
        return results;
    }
    
    /**
     * Invalidate the MCC
     * @param containerIds the container ids
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> invalidateMCCForCVFU(List<String> containerIds)
    {
        return _doMCCAction(containerIds, "ODF_Pilotage_CFVU_MCC_Validated_Rights", container -> _mccWorkflowHelper.invalidateMCCForCVFU(container));
    }
    
    private Map<String, Object> _doMCCAction(List<String> containerIds, String rightId, Function<Container, Boolean> mccWorkflowAction)
    {
        Map<String, Object> results = new HashMap<>();
        
        List<Map<String, Object>> noRightContents = new ArrayList<>();
        List<Map<String, Object>> lockedContents = new ArrayList<>();
        List<Map<String, Object>> workflowErrorContents = new ArrayList<>();
        List<String> successContents = new ArrayList<>();
        
        UserIdentity user = _currentUserProvider.getUser();
        
        for (String containerId : containerIds)
        {
            Container container = _resolver.resolveById(containerId);
            
            try
            {
                if (_rightManager.hasRight(user, rightId, container) != RightResult.RIGHT_ALLOW)
                {
                    noRightContents.add(Map.of("id", containerId, "title", container.getTitle()));
                }
                else if (container.isLocked() && !LockHelper.isLockOwner(container, user))
                {
                    lockedContents.add(Map.of("id", containerId, "title", container.getTitle()));
                }
                else
                {
                    mccWorkflowAction.apply(container);
                    successContents.add(containerId);
                }
            }
            catch (MCCWorkflowException e)
            {
                workflowErrorContents.add(Map.of("id", containerId, "title", container.getTitle(), "errorType", e.getType().name()));
            }
        }
        
        results.put("success-contents", successContents);
        results.put("noright-contents", noRightContents);
        results.put("locked-contents", lockedContents);
        results.put("invalidmccstatus-contents", workflowErrorContents);
        
        return results;
        
    }
}
