/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.cost;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.cost.entity.CostComputationData;
import org.ametys.plugins.odfpilotage.cost.entity.CostComputationDataCache;
import org.ametys.plugins.odfpilotage.cost.entity.Effectives;
import org.ametys.plugins.odfpilotage.cost.entity.EqTD;
import org.ametys.plugins.odfpilotage.cost.entity.Groups;
import org.ametys.plugins.odfpilotage.cost.entity.NormDetails;
import org.ametys.plugins.odfpilotage.cost.entity.OverriddenData;
import org.ametys.plugins.odfpilotage.cost.entity.ProgramItemData;
import org.ametys.plugins.odfpilotage.cost.entity.VolumesOfHours;
import org.ametys.plugins.odfpilotage.cost.eqtd.EqTDComputationMode;
import org.ametys.plugins.odfpilotage.cost.eqtd.EqTDComputationModeExtensionPoint;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper.StepHolderStatus;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * This component computes values used by the cost modeling tool and some reports (synthesis, coût des maquettes, potentiel enseignant).
 */

public class CostComputationComponent extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role name */
    public static final String ROLE = CostComputationComponent.class.getName();

    private static final String __EFFECTIF_ESTIMATED = "numberOfStudentsEstimated";
    
    /** The ODF enumeration helper */
    private OdfReferenceTableHelper _refTableHelper;
    
    /** The ODF pilotage helper */
    private PilotageHelper _pilotageHelper;
    
    /** The ODF helper */
    private ODFHelper _odfHelper;
    
    /** The TD equivalent computation mode extension point */
    private EqTDComputationModeExtensionPoint _eqTDComputationModeEP;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _eqTDComputationModeEP = (EqTDComputationModeExtensionPoint) manager.lookup(EqTDComputationModeExtensionPoint.ROLE);
    }
    
    /**
     * Call methods to initialize data and compute groups distribution from an orgUnit
     * @param orgUnit the orgUnit
     * @param catalog the catalog
     * @param lang the lang
     * @param needsAdditionalComputation <code>true</code> to calculate effectives and eqTD from mutualizations, only need for some uses
     * @return CostData object containing informations about the program
     */
    public CostComputationData computeCostsOnOrgUnit(OrgUnit orgUnit, String catalog, String lang, boolean needsAdditionalComputation)
    {
        return computeCostsOnOrgUnit(orgUnit, catalog, lang, needsAdditionalComputation, new OverriddenData());
    }
    
    /**
     * Call methods to initialize data and compute groups distribution from an orgUnit
     * @param orgUnit the orgUnit
     * @param catalog the catalog
     * @param lang the lang
     * @param needsAdditionalComputation <code>true</code> to calculate effectives and eqTD from mutualizations, only need for some uses
     * @param overriddenData overridden data by the user
     * @return CostData object containing informations about the program
     */
    public CostComputationData computeCostsOnOrgUnit(OrgUnit orgUnit, String catalog, String lang, boolean needsAdditionalComputation, OverriddenData overriddenData)
    {
        List<Program> programs = _odfHelper.getProgramsFromOrgUnit(orgUnit, catalog, lang);
        CostComputationDataCache cache = _computeCostsOnPrograms(programs, needsAdditionalComputation, overriddenData);
        
        // Compute cumulative values for the orgunit
        Effectives effectives = new Effectives();
        VolumesOfHours volumesOfHours = new VolumesOfHours();
        EqTD eqTD = new EqTD();
        Double localEqTD = 0D;
        Double proratedEqTD = 0D;
        Double localEffective = 0D;
        
        for (Program program : programs)
        {
            String programId = program.getId();
            
            // Effectives
            localEffective += Optional.of(programId)
                    .map(cache::getEffectives)
                    .map(eff -> eff.getLocalEffective(program.getName()))
                    .orElse(0d);
            
            // Volume of hours
            volumesOfHours.sum(cache.getVolumesOfHours(programId), 1D);
            
            // EqTD
            EqTD programEqTD = cache.getEqTD(programId);
            eqTD.sum(programEqTD);
            localEqTD += programEqTD.getLocalEqTD()
                    .values()
                    .stream()
                    .reduce(0D, Double::sum);
            proratedEqTD += programEqTD.getProratedEqTD()
                    .values()
                    .stream()
                    .reduce(0D, Double::sum);
        }
        
        String orgUnitName = orgUnit.getName();
        effectives.addLocalEffective(orgUnitName, localEffective);
        eqTD.addLocalEqTD(orgUnitName, localEqTD);
        eqTD.addProratedEqTD(orgUnitName, proratedEqTD);
        
        // Add to cache
        String orgUnitId = orgUnit.getId();
        cache.putEffectives(orgUnitId, effectives);
        cache.putVolumesOfHours(orgUnitId, volumesOfHours);
        cache.putEqTD(orgUnitId, eqTD);
        
        return cache.getCostComputationData();
    }
    
    /**
     * Call methods to initialize data and compute groups distribution from a catalog
     * @param programs the catalog
     * @param needsAdditionalComputation <code>true</code> to calculate effectives and eqTD from mutualizations, only need for some uses
     * @return CostData object containing informations about the program
     */
    public CostComputationData computeCostsOnPrograms(List<Program> programs, boolean needsAdditionalComputation)
    {
        return computeCostsOnPrograms(programs, needsAdditionalComputation, new OverriddenData());
    }
    
    /**
     * Call methods to initialize data and compute groups distribution from a catalog
     * @param programs the catalog
     * @param needsAdditionalComputation <code>true</code> to calculate effectives and eqTD from mutualizations, only need for some uses
     * @param overriddenData overridden data by the user
     * @return CostData object containing informations about the program
     */
    public CostComputationData computeCostsOnPrograms(List<Program> programs, boolean needsAdditionalComputation, OverriddenData overriddenData)
    {
        return _computeCostsOnPrograms(programs, needsAdditionalComputation, overriddenData).getCostComputationData();
    }
    
    /**
     * Call methods to initialize data and compute groups distribution from a catalog
     * @param programs the catalog
     * @param needsAdditionalComputation <code>true</code> to calculate effectives and eqTD from mutualizations, only need for some uses
     * @param overriddenData overridden data by the user
     * @return CostData object containing informations about the program
     */
    protected CostComputationDataCache _computeCostsOnPrograms(List<Program> programs, boolean needsAdditionalComputation, OverriddenData overriddenData)
    {
        CostComputationDataCache cache = new CostComputationDataCache(
            programs,
            _refTableHelper.getItems(PilotageHelper.NORME),
            _refTableHelper.getItems(OdfReferenceTableHelper.ENSEIGNEMENT_NATURE),
            _pilotageHelper.getYearId().orElse(null),
            overriddenData
        );
        
        for (Program program : programs)
        {
            _searchAndComputeCourseParts(program, cache);
            // Separated compute because we need to explore all the children
            _computeLocalValues(program, program.getName(), null, null, cache);
        }
        
        if (needsAdditionalComputation)
        {
            // For each program item data, check if local effectives are computed for each path
            // Can happen if some elements in the structure are shared with programs not calculated in the current computation
            CostComputationData costData = cache.getCostComputationData();
            for (String itemId : costData.keySet())
            {
                ProgramItemData itemData = costData.get(itemId);
                if (itemData.isCoursePart())
                {
                    Effectives effectives = itemData.getEffectives();
                    for (String path : itemData.getPathes())
                    {
                        // Compute only missing data (needs for ventilation)
                        if (effectives.getLocalEffective(path) == null)
                        {
                            // Compute local effectives for each course part
                            // We need the parent effectives to do the computation
                            // That's why we iterate on each element of the tree
                            String parentId = null;
                            String currentPath = "";
                            for (Content contentInPath : _pilotageHelper.getContentsFromPath(path).toList())
                            {
                                currentPath += contentInPath.getName();
                                String contentInPathId = contentInPath.getId();
                                Supplier<Double> weightSupplier = () -> contentInPath instanceof Program programItem ? _getWeight(programItem, cache) : 1d;
                                _computeLocalEffective(contentInPathId, currentPath, parentId, weightSupplier, cache);
                                currentPath += ModelItem.ITEM_PATH_SEPARATOR;
                            }
                            
                            // Compute proprated eqTD for each path of the course part
                            _computeProratedEqTD(itemId, path, cache);
                        }
                    }
                }
            }
        }
        
        // Sum local eqTD in cache, not possible before because of counting occurrences of each course part
        for (Program program : programs)
        {
            _localEqTD(program, program.getName(), cache);
        }
        
        return cache;
    }
    
    private void _computeLocalValues(ProgramItem programItem, String currentPath, String parentProgramItemId, Container currentStep, CostComputationDataCache cache)
    {
        String programItemId = programItem.getId();
        
        // If it has not been explored yet for this path
        if (_computeLocalEffective(programItemId, currentPath, parentProgramItemId, () -> _getWeight(programItem, cache), cache))
        {
            Double proratedEqTD = 0d;
            
            if (programItem instanceof Course course && !course.hasCourseLists())
            {
                for (CoursePart coursePart : course.getCourseParts())
                {
                    String coursePartId = coursePart.getId();
                    String coursePartPath = currentPath + ModelItem.ITEM_PATH_SEPARATOR + coursePart.getName();
                    _computeLocalEffective(coursePartId, coursePartPath, programItemId, () -> 1d, cache);
                    _computeLocalEqTD(coursePart, course, coursePartPath, currentStep, cache);
                    proratedEqTD += _computeProratedEqTD(coursePartId, coursePartPath, cache);
                }
            }
            else
            {
                Container newCurrentStep = _pilotageHelper.isContainerOfNature(programItem, cache.getYearNature())
                        ? (Container) programItem
                        : currentStep;
                
                // Cumulate children local effectives if we are on top level of a step and effectives are not defined
                Effectives effectives = cache.getEffectives(programItemId);
                boolean cumulateLocalEffectiveBySum = currentStep == null && effectives.getEstimatedEffectiveByPath().isEmpty();
                Double cumulatedLocalEffective = 0d;
                
                List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
                for (ProgramItem child : children)
                {
                    String childPath = currentPath + ModelItem.ITEM_PATH_SEPARATOR + child.getName();
                    _computeLocalValues(child, childPath, programItemId, newCurrentStep, cache);
                    proratedEqTD += cache.getEqTD(child.getId()).getProratedEqTD(childPath);
                    if (cumulateLocalEffectiveBySum)
                    {
                        cumulatedLocalEffective += Optional.of(child)
                                .map(ProgramItem::getId)
                                .map(cache::getEffectives)
                                .map(eff -> eff.getLocalEffective(childPath))
                                .orElse(0d);
                    }
                }
                
                // Sum children on local effectives if we are on top level of a step and effectives are not defined
                if (cumulateLocalEffectiveBySum)
                {
                    effectives.addLocalEffective(currentPath, cumulatedLocalEffective);
                    cache.putEffectives(programItemId, effectives);
                }
            }
            
            _addProratedEqTD(programItemId, currentPath, proratedEqTD, cache);
        }
    }
    
    private Double _computeProratedEqTD(String programItemId, String currentPath, CostComputationDataCache cache)
    {
        Effectives effectives = cache.getEffectives(programItemId);
        Double localEffectives = effectives.getLocalEffective(currentPath);
        if (localEffectives != null)
        {
            Double ratioGlobalEffectives = effectives.getEstimatedEffective()
                    .filter(d -> d != 0D)
                    .map(d -> localEffectives / d)
                    .orElse(0D);
            
            EqTD eqTD = cache.getEqTD(programItemId);
            eqTD.addProratedEqTD(currentPath, eqTD.getGlobalEqTD() * ratioGlobalEffectives);
            cache.putEqTD(programItemId, eqTD);
            
            return eqTD.getProratedEqTD(currentPath);
        }
        
        return 0d;
    }
    
    private void _addProratedEqTD(String programItemId, String currentPath, Double proratedEqTD, CostComputationDataCache cache)
    {
        EqTD eqTD = cache.getEqTD(programItemId);
        eqTD.addProratedEqTD(currentPath, proratedEqTD);
        cache.putEqTD(programItemId, eqTD);
    }
    
    private boolean _computeLocalEffective(String programItemId, String currentPath, String parentProgramItemId, Supplier<Double> weightSupplier, CostComputationDataCache cache)
    {
        Effectives effectives = cache.getEffectives(programItemId);
        
        // Value already in the cache
        if (effectives.getLocalEffective(currentPath) != null)
        {
            return false;
        }
        
        // Compute the locale effective:
        // Case 1: Overridden effective / number of pathes for the current item
        effectives.getOverriddenEffective()
            // Case 2: Entered effective / number of pathes for the current item
            .or(effectives::getEnteredEffective)
            .map(eff -> eff / _getNbPathes(programItemId, cache))
            // Case 3: Parent local effective * weight of the current element
            .or(() -> _computeLocalEffectiveFromParent(currentPath, parentProgramItemId, weightSupplier, cache))
            // Then add it to the cache
            .ifPresent(localEffective -> effectives.addLocalEffective(currentPath, localEffective));
        
        cache.putEffectives(programItemId, effectives);
        
        return true;
    }
    
    private Integer _getNbPathes(String programItemId, CostComputationDataCache cache)
    {
        return Optional.of(programItemId)
                .map(cache::getPathes)
                .map(Set::size)
                .orElse(1); // Prevent from dividing by 0
    }
    
    private Optional<Double> _computeLocalEffectiveFromParent(String currentPath, String parentProgramItemId, Supplier<Double> weightSupplier, CostComputationDataCache cache)
    {
        String parentPath = StringUtils.substringBeforeLast(currentPath, ModelItem.ITEM_PATH_SEPARATOR);
        return Optional.ofNullable(parentProgramItemId)
            .map(cache::getEffectives)
            .map(eff -> eff.getLocalEffective(parentPath))
            .map(eff -> eff * weightSupplier.get());
    }
    
    private void _computeLocalEqTD(CoursePart coursePart, Course currentParentCourse, String currentPath, Container currentStep, CostComputationDataCache cache)
    {
        EqTD coursePartEqTD = cache.getEqTD(coursePart.getId());
        coursePartEqTD.addLocalEqTD4CoursePart(currentPath, _isHeld(coursePart, currentParentCourse, currentStep));
        cache.putEqTD(coursePart.getId(), coursePartEqTD);
    }
    
    private Double _localEqTD(ProgramItem programItem, String currentPath, CostComputationDataCache cache)
    {
        Double localEqTDsum = 0D;
        if (programItem instanceof Course course && !course.hasCourseLists())
        {
            for (CoursePart coursePart : course.getCourseParts())
            {
                EqTD coursePartEqTD = cache.getEqTD(coursePart.getId());
                localEqTDsum += coursePartEqTD.getLocalEqTD(currentPath + ModelItem.ITEM_PATH_SEPARATOR + coursePart.getName());
            }
        }
        else
        {
            for (ProgramItem child : _odfHelper.getChildProgramItems(programItem))
            {
                localEqTDsum += _localEqTD(child, currentPath + ModelItem.ITEM_PATH_SEPARATOR + child.getName(), cache);
            }
        }
        
        EqTD eqTD = cache.getEqTD(programItem.getId());
        eqTD.addLocalEqTD(currentPath, localEqTDsum);
        cache.putEqTD(programItem.getId(), eqTD);
        
        return localEqTDsum;
    }
    
    private boolean _isHeld(CoursePart coursePart, Course currentCourse, Container currentStep)
    {
        Course courseHolder = coursePart.getCourseHolder();
        if (courseHolder == null)
        {
            return true;
        }
        
        if (currentCourse.equals(courseHolder))
        {
            Container stepHolder = _getStepHolder(courseHolder);
            return stepHolder == null || stepHolder.equals(currentStep);
        }
        
        return false;
    }
    
    /**
     * Call methods to initialize data and compute groups distribution from a program
     * @param program the program
     * @param needsAdditionalComputation <code>true</code> to calculate effectives and eqTD from mutualizations, only need for some uses
     * @return CostData object containing informations about the program
     */
    public CostComputationData computeCostsOnProgram(Program program, boolean needsAdditionalComputation)
    {
        return computeCostsOnProgram(program, needsAdditionalComputation, new OverriddenData());
    }
    
    /**
     * Call methods to initialize data and compute groups distribution from a program
     * @param program the program
     * @param needsAdditionalComputation <code>true</code> to calculate effectives and eqTD from mutualizations, only need for some uses
     * @param overriddenData overridden data by the user
     * @return CostData object containing informations about the program
     */
    public CostComputationData computeCostsOnProgram(Program program, boolean needsAdditionalComputation, OverriddenData overriddenData)
    {
        return computeCostsOnPrograms(List.of(program), needsAdditionalComputation, overriddenData);
    }
    
    /**
     * Iterate the program item structure and compute the global effective for each course part
     * @param programItem the program item
     * @param cache the cache values
     */
    private void _searchAndComputeCourseParts(ProgramItem programItem, CostComputationDataCache cache)
    {
        String programItemId = programItem.getId();
        
        // If has been explorer, pass...
        if (cache.addExploredItem(programItemId))
        {
            _computePathes(programItem, cache);
            
            // Compute effectives
            _computeEffectives(programItem, cache);
            
            if (programItem instanceof Course course && !course.hasCourseLists())
            {
                _computeCourseParts(course, cache);
            }
            else
            {
                double weight = _getWeight(programItem, cache);
                
                VolumesOfHours volumesOfHours = new VolumesOfHours();
                EqTD eqTD = new EqTD();
                List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
                for (ProgramItem child : children)
                {
                    // Recursive call to continue exploration of the structure
                    _searchAndComputeCourseParts(child, cache);
                    
                    ProgramItemData childData = cache.getProgramItemData(child.getId());
                    
                    if (childData != null)
                    {
                        // Add volumes
                        Optional.ofNullable(childData.getVolumesOfHours())
                            .ifPresent(childVolumesOfHours -> volumesOfHours.sum(childVolumesOfHours, weight));
                        
                        // Add EqTD
                        Optional.ofNullable(childData.getEqTD())
                            .ifPresent(childEqTD -> eqTD.sum(childEqTD));
                    }
                }
                
                // Update the cache
                cache.putVolumesOfHours(programItemId, volumesOfHours);
                cache.putEqTD(programItemId, eqTD);
            }
        }
    }
    
    private Set<String> _computePathes(ProgramItem programItem, CostComputationDataCache cache)
    {
        Set<String> pathes = cache.getPathes(programItem.getId());
        
        if (pathes == null)
        {
            String programItemName = programItem.getName();
            pathes = programItem instanceof Program
                // Returns program-licence-chimie
                ? Set.of(programItemName)
                // Returns:
                //  - program-licence-chimie/subprogram-parcours-chimie-moleculaire/container-annee-1-chimie/container-semestre-1
                //  - program-licence-chimie/subprogram-parcours-biochimie/container-annee-1-biochimie/container-semestre-1
                : _odfHelper.getParentProgramItems(programItem)
                    // Returns:
                    // - container-annee-1-chimie
                    // - container-annee-1-biochimie
                    .stream()
                    // Returns:
                    //  - program-licence-chimie/subprogram-parcours-chimie-moleculaire/container-annee-1-chimie
                    //  - program-licence-chimie/subprogram-parcours-biochimie/container-annee-1-biochimie
                    .map(parent -> _computePathes(parent, cache))
                    .flatMap(Set::stream)
                    .distinct()
                    // Contatenate current program item (container-semestre-1) and parent pathes
                    .map(parentPath -> parentPath + ModelItem.ITEM_PATH_SEPARATOR + programItemName)
                    .collect(Collectors.toSet());
            
            cache.putPathes(programItem.getId(), pathes);
        }
        
        return pathes;
    }
    
    /**
     * Compute course parts of a course if it hasn't been already computed.
     * @param course The parent course
     * @param cache the cache values
     */
    private void _computeCourseParts(Course course, CostComputationDataCache cache)
    {
        VolumesOfHours volumesOfHours = new VolumesOfHours();
        EqTD eqTD = new EqTD();
        
        // We compute effectives, volumes of hours and td equivalents for each course part
        for (CoursePart coursePart : course.getCourseParts())
        {
            _computeCoursePart(coursePart, cache);
            
            String coursePartId = coursePart.getId();
            
            // Volumes of hours
            Optional.of(coursePartId)
                .map(cache::getVolumesOfHours)
                .ifPresent(childVolumesOfHours -> volumesOfHours.sum(childVolumesOfHours, 1D));
            
            // EqTD
            Optional.of(coursePartId)
                .map(cache::getEqTD)
                .ifPresent(childEqTD -> eqTD.sum(childEqTD));
        }
        
        String courseId = course.getId();
        
        cache.putVolumesOfHours(courseId, volumesOfHours);
        cache.putEqTD(courseId, eqTD);
    }
    
    /**
     * Get the step holder associated to a program item
     * @param programItem the program item
     * @param <T> The type of the program item to search on, should extends {@link Content} and {@link ProgramItem}
     * @return the step holder
     */
    private <T extends Content & ProgramItem> Container _getStepHolder(T programItem)
    {
        Pair<StepHolderStatus, Container> stepHolder = _pilotageHelper.getStepHolder(programItem);
        switch (stepHolder.getKey())
        {
            case NO_YEAR:
                getLogger().warn("[{}][{}] Impossible de trouver une nature de conteneur 'annee'.", programItem.getTitle(), programItem.getCode());
                break;
            case NONE:
                getLogger().info("[{}][{}] Aucune année n'est rattachée à l'ELP '{}' directement ou indirectement.", programItem.getTitle(), programItem.getCode(), programItem.getTitle());
                break;
            case MULTIPLE:
                getLogger().info("[{}][{}] Plusieurs années sont rattachées à l'ELP '{}', impossible de déterminer laquelle est porteuse.", programItem.getTitle(), programItem.getCode(), programItem.getTitle());
                break;
            case WRONG_YEAR:
                getLogger().info("[{}][{}] L'année porteuse {} n'est pas dans l'arborescence de l'ELP '{}'.", programItem.getTitle(), programItem.getCode(), stepHolder.getValue().getTitle(), programItem.getTitle());
                break;
            default:
                // Nothing to do
                break;
        }
        
        return stepHolder.getValue();
    }
    
    /**
     * Get all informations about the norm
     * @param coursePart informations about the teaching hours of a course
     * @return norm details
     * @param cache the cache values
     */
    private NormDetails _computeNormDetails(CoursePart coursePart, CostComputationDataCache cache)
    {
        String nature = coursePart.getNature();
        return _getNorm(coursePart, nature, cache)                              // Get the norm on the course part
                .or(() -> _getNorm(_getStepHolder(coursePart), nature, cache))  // Or on the step holder if defined
                .map(n -> cache.getNormDetailsForNature(n, nature))             // Get norm details from the norm
                .orElseGet(() -> cache.getEffectiveMinMaxByNature(nature));     // Or get norm details from the nature
    }
    
    private Container _getStepHolder(CoursePart coursePart)
    {
        // Course holder
        Course courseHolder = coursePart.getCourseHolder();
        if (courseHolder == null)
        {
            getLogger().warn("[{}][{}] L'ELP porteur est soit vide, soit invalide.", coursePart.getTitle(), coursePart.getCode());
        }
        
        // Norm details (effectives max and min sup)
        return Optional.ofNullable(courseHolder)
            .map(this::_getStepHolder)
            .orElse(null);
    }
   
    /**
     * Get the computed global and entered effectives
     * @param coursePart the course part
     * @param cache the cached values
     */
    private void _computeCoursePart(CoursePart coursePart, CostComputationDataCache cache)
    {
        String coursePartId = coursePart.getId();

        // If has been explorer, pass...
        if (cache.addExploredItem(coursePartId))
        {
            String coursePartNature = coursePart.getNature();
            
            // Set it is a course part
            cache.setIsCoursePart(coursePartId, true);
            
            // Volume of hours
            VolumesOfHours volumesOfHours = new VolumesOfHours();
            volumesOfHours.addVolume(coursePartNature, coursePart.getNumberOfHours());
            Double overriddenVolumeOfHours = cache.getOverriddenVolumeOfHours(coursePartId);
            if (overriddenVolumeOfHours != null)
            {
                volumesOfHours.addOverriddenVolume(coursePartNature, overriddenVolumeOfHours);
            }
            cache.putVolumesOfHours(coursePartId, volumesOfHours);

            // Effectives and pathes
            Effectives effectives = new Effectives();
            List<Course> parentCourses = coursePart.getCourses();
            effectives.setNbOccurrences(parentCourses.size());
            Set<String> pathes = new HashSet<>();
            for (Course course : parentCourses)
            {
                String courseName = course.getName();
                effectives.add(_computeEffectives(course, cache), courseName);
                pathes.addAll(
                    _computePathes(course, cache)
                        .stream()
                        .map(parentPath -> parentPath + ModelItem.ITEM_PATH_SEPARATOR + courseName)
                        .collect(Collectors.toSet())
                );
            }
            cache.putEffectives(coursePartId, effectives);
            cache.putPathes(coursePartId, pathes);
            
            // Norm details (effectives max and min sup)
            NormDetails normDetails =  _computeNormDetails(coursePart, cache);
            cache.putNormDetails(coursePartId, normDetails);
            if (normDetails == null)
            {
                getLogger().warn("[{}][{}] La nature d'enseignement est soit vide, soit invalide.", coursePart.getTitle(), coursePart.getCode());
                normDetails = new NormDetails(null, null);
            }
            
            // Groups
            Double globalEffectives = effectives.getEstimatedEffective().orElse(0d);
            Long computedGroups = _computeGroups(globalEffectives, normDetails);
            Groups groups = new Groups();
            groups.setOverriddenGroups(cache.getOverriddenGroups(coursePartId));
            groups.setGroupsToOpen(coursePart.getValue("groupsToOpen"));
            groups.setComputedGroups(computedGroups);
            cache.putGroups(coursePartId, groups);
            
            // eqTD
            ProgramItemData programItemData = cache.getProgramItemData(coursePartId);
            Double eqTDCoef = Optional.of(coursePart)
                    .map(cp -> cp.<String>getValue("eqTD"))
                    .map(PilotageHelper::transformEqTD2Double)
                    .orElseGet(() -> cache.getEqTDByNature(coursePartNature));
            EqTDComputationMode eqTDComputationMode = _eqTDComputationModeEP.getExtension(cache.getEqTDComputationByNature(coursePartNature));
            Double globalEqTD = eqTDComputationMode.computeEqTD(programItemData, eqTDCoef);
            EqTD eqTD = new EqTD();
            eqTD.addGlobalEqTD(coursePartId, globalEqTD);
            cache.putEqTD(coursePartId, eqTD);
        }
    }
    

    
    /**
     * Compute the global and entered effectives
     * @param programItem the item we want to evaluate the capacity
     * @param cache the cached values
     * @return the computed effective
     */
    private Effectives _computeEffectives(ProgramItem programItem, CostComputationDataCache cache)
    {
        String programItemId = programItem.getId();
        
        Effectives effectives = cache.getEffectives(programItemId);
        if (effectives == null)
        {
            effectives = new Effectives();

            List<ProgramItem> programItemParents = _odfHelper.getParentProgramItems(programItem);
            if (!(programItem instanceof Program))
            {
                effectives.setNbOccurrences(programItemParents.size());
            }
            
            Optional<Double> overriddenEffective = Optional.of(programItemId)
                    .map(cache::getOverriddenEffective)
                    .map(Long::doubleValue);
            
            effectives.setOverriddenEffective(overriddenEffective);
            
            Optional<Double> enteredEffective = Optional.of(programItem)
                    .filter(Predicate.not(CourseList.class::isInstance))
                    .map(Content.class::cast)
                    .map(pi -> pi.<Long>getValue(__EFFECTIF_ESTIMATED))
                    .map(Long::doubleValue);

            // Written effective (optional)
            effectives.setEnteredEffective(enteredEffective);

            boolean isYear = _pilotageHelper.isContainerOfNature(programItem, cache.getYearNature());
            
            Optional<Double> writtenEffective = overriddenEffective.or(() -> enteredEffective);
            if (writtenEffective.isPresent())
            {
                Double eff = writtenEffective.get();
                effectives.addEstimatedEffective(programItem.getName(), eff);
                if (isYear)
                {
                    effectives.addComputedEffective(programItemId, eff);
                }
            }
            // If there is no written (overridden or entered) effective for year
            else if (isYear)
            {
                getLogger().warn("[{}][{}] Aucun effectif n'a été saisi sur cette année.", ((Container) programItem).getTitle(), programItem.getCode());
            }
            
            for (ProgramItem parent : programItemParents)
            {
                effectives.add(_computeEffectives(parent, cache), parent.getName());
            }
            
            cache.putEffectives(programItemId, effectives);
        }
        return effectives.cloneWithWeight(_getWeight(programItem, cache));
    }
    
    /**
     * Define the weight of an item
     * @param programItem the item to evaluate
     * @param cache the cache
     * @return the weight
     */
    private double _getWeight(ProgramItem programItem, CostComputationDataCache cache)
    {
        Double weight = cache.getWeight(programItem.getId());
        
        if (weight == null)
        {
            weight = 1D;
            if (programItem instanceof CourseList courseList)
            {
                ChoiceType type = courseList.getType();
                
                switch (type)
                {
                    case OPTIONAL:
                        // If it's an optional list, weight is zero
                        weight = 0D;
                        break;
                    case CHOICE:
                        // If it's a choice list, the weight is computed with min and course list size
                        long min = courseList.getMinNumberOfCourses();
                        long max = courseList.getMaxNumberOfCourses();
                        long size = courseList.getCourses().size();
                        
                        if (min != max)
                        {
                            getLogger().warn("[{}][{}] La liste contient min={} et max={}. Retenu {}", courseList.getTitle(), courseList.getCode(), min, max, min);
                        }
                        
                        if (min > size)
                        {
                            getLogger().warn("[{}][{}] La liste contient min={} mais n'a que {} éléments.", courseList.getTitle(), courseList.getCode(), min, size);
                        }
                        else
                        {
                            weight = (double) min / (double) size;
                        }
                        break;
                    default:
                        // If it's a mandatory list (or null type), the weight is 1
                        break;
                }
            }
            
            cache.putWeight(programItem.getId(), weight);
        }
        return weight;
    }
    
    /**
     * Compute the number of groups needed
     * @param effective the forecast capacity
     * @param normDetails all informations needed to compute the norm
     * @return the number of groups to open
     */
    private Long _computeGroups(Double effective, NormDetails normDetails)
    {
        long effectiveMax = normDetails.getEffectiveMaxOrDefault();
        long effectiveMinSup = normDetails.getEffectiveMinSupOrDefault();
        if (effective == 0D)
        {
            return 0L;
        }
        
        if (effective < effectiveMax)
        {
            return 1L;
        }
        
        Long groups = Math.round(effective) / effectiveMax;
        if (effective - groups * effectiveMax >= effectiveMinSup)
        {
            groups++;
        }
        
        return groups;
    }
    
    /**
     * Retrieve the norm value of a content
     * @param content the content
     * @param nature the nature of the content
     * @param cache the cache values
     * @return the norm of the content
     */
    private Optional<String> _getNorm(Content content, String nature, CostComputationDataCache cache)
    {
        // The content can be null for step holder
        if (content == null)
        {
            return Optional.empty();
        }
        
        String logName = content instanceof Container ? "année porteuse" : "heure d'enseignement";
        
        ContentValue normValue = content.getValue("norme");
        Optional<String> normIdOpt = Optional.ofNullable(normValue)
                                 .map(ContentValue::getContentId)
                                 .filter(StringUtils::isNotEmpty);
        
        if (normIdOpt.isPresent())
        {
            String normId = normIdOpt.get();
            if (!cache.normExists(normId))
            {
                getLogger().warn("[{}][{}] L'{} '{}' possède une norme invalide.", content.getTitle(), content.getValue("code"), logName, content.getTitle());
            }
            else if (!cache.natureInNormExists(normId, nature))
            {
                if (getLogger().isWarnEnabled())
                {
                    Content norm = normValue.getContent();
                    getLogger().warn("[{}][{}] La norme '{}' n'est pas définie pour la nature d'enseignement {}.", content.getTitle(), content.getValue("code"), norm.getTitle(), _refTableHelper.getItemCode(nature));
                }
            }
            else
            {
                return normIdOpt;
            }
        }
        else if (getLogger().isInfoEnabled())
        {
            getLogger().info("[{}][{}] L'{} '{}' n'a pas de norme associée.", content.getTitle(), content.getValue("code"), logName, content.getTitle());
        }
        
        return Optional.empty();
    }
}
