/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.helper;

import java.time.LocalDate;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.cache.Cache;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowException.ExceptionType;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.workflow.ValidateProgramItemTreeCondition;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareComposite;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;

/**
 * Helper for MCC workflow
 */
public class MCCWorkflowHelper extends AbstractWorkflowHelper implements Initializable
{
    /** The component role. */
    public static final String ROLE = MCCWorkflowHelper.class.getName();
    
    /** The super right for rules validation state on year */
    public static final String RULES_VALIDATED_SUPER_RIGHT_ID = "ODF_Pilotage_Rules_Validated_Super_Rights";
    
    /** The super right for MCC fields validation state on year */
    public static final String MCC_VALIDATED_SUPER_RIGHT_ID = "ODF_Pilotage_MCC_Validated_Super_Rights";
    
    /** The super right for MCC orgunit validation state */
    public static final String MCC_ORGUNIT_VALIDATED_SUPER_RIGHT_ID = "ODF_Pilotage_MCC_Orgunit_Validated_Super_Rights";
    
    /** The super right for MCC CFVU validation state */
    public static final String MCC_CFVU_VALIDATED_SUPER_RIGHT_ID = "ODF_Pilotage_MCC_CFVU_Validated_Super_Rights";
    
    /** The attribute name for the MCC validated pdf repeater */
    public static final String MCC_VALIDATED_PDF_REPEATER = "mcc-validated-pdf";
    
    /** The identifier for the MCC workflow action */
    public static final Integer MCC_WORKFLOW_ACTION_ID = 222222;
    
    /** The attribute name for the pilotage composite */
    private static final String __MCC_WORKFLOW_COMPOSITE = "mcc_workflow";
    
    /** The suffixe for status of MCC workflow step */
    private static final String __STATUS_SUFFIX = "_status";
    
    /** The prefix for rules mention validation attributes */
    private static final String __MCC_VALIDATION_PREFIX = "mcc_validation";
    
    /** The prefix for rules mention validation attributes */
    private static final String __RULES_VALIDATION_PREFIX = "rules_validation";
    
    /** The prefix for MCC orgunit validation attributes */
    private static final String __MCC_ORGUNIT_VALIDATION_PREFIX = "mcc_orgunit_validation";
    
    /** The prefix for MCC CFVU validation attributes */
    private static final String __CFVU_MCC_VALIDATION_PREFIX = "cfvu_mcc_validation";
    
    private static final String __PARENT_CONTAINERS_WITH_HIGHER_STATUS_CACHE_ID = MCCWorkflowHelper.class.getName() + "$parentContainers";
    
    private static final String __PARENT_CONTAINERS_WITH_HIGHER_RULES_STATUS_CACHE_ID = MCCWorkflowHelper.class.getName() + "$parentContainersForRules";
    
    private static final String __PARENT_CONTAINERS_WITH_HIGHER_MCC_STATUS_CACHE_ID = MCCWorkflowHelper.class.getName() + "$parentContainersForMCC";

    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createRequestCache(__PARENT_CONTAINERS_WITH_HIGHER_STATUS_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_CFVU_STATUS_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_CFVU_STATUS_DESCRIPTION"),
                true);
        
        _cacheManager.createRequestCache(__PARENT_CONTAINERS_WITH_HIGHER_RULES_STATUS_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_RULES_STATUS_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_RULES_STATUS_DESCRIPTION"),
                true);
        
        _cacheManager.createRequestCache(__PARENT_CONTAINERS_WITH_HIGHER_MCC_STATUS_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_STATUS_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_CONTAINER_WITH_HIGHER_MCC_STATUS_DESCRIPTION"),
                true);
    }
    
    private Cache<String, String> _getContainerParentsCache()
    {
        return _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_STATUS_CACHE_ID);
    }
    
    private Cache<String, String> _getContainerParentsCacheForRules()
    {
        return _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_RULES_STATUS_CACHE_ID);
    }
    
    private Cache<String, String> _getContainerParentsCacheForMCC()
    {
        return _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_MCC_STATUS_CACHE_ID);
    }
    
    /**
     * Invalidate all caches
     */
    public void invalidateAllCaches()
    {
        _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_STATUS_CACHE_ID).invalidateAll();
        _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_RULES_STATUS_CACHE_ID).invalidateAll();
        _cacheManager.get(__PARENT_CONTAINERS_WITH_HIGHER_MCC_STATUS_CACHE_ID).invalidateAll();
    }
    
    /**
     * Enumeration for steps of MCC workflow
     *
     */
    public enum MCCWorkflowStep
    {
        /** Rules are validated to mention level */
        RULES_VALIDATED,
        /** MCC are validated to mention level */
        MCC_VALIDATED,
        /** MCC are validated to orgunit level */
        MCC_ORGUNIT_VALIDATED,
        /** MCC are CFVU validated */
        CFVU_MCC_VALIDATED
    }
    
    /**
     * Get the first parent container with the higher MCC status
     * @param programItem the program item
     * @return the first parent program with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatus(ProgramItem programItem)
    {
        Cache<String, String> cache = _getContainerParentsCache();
        if (cache.hasKey(programItem.getId()))
        {
            String parentId = cache.get(programItem.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container parentContainer = _computeParentContainerWithHigherMCCStatus(programItem);
            cache.put(programItem.getId(), parentContainer != null ? parentContainer.getId() : null);
            return parentContainer;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatus(ProgramItem programItem)
    {
        Set<Container> parentContainers = _getParentYearContainers(programItem);
        return parentContainers.stream()
            .sorted(new MCCStatusComparator().reversed())
            .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent program with the higher MCC status
     * @param coursePart the course part
     * @return the first parent program with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatus(CoursePart coursePart)
    {
        Cache<String, String> cache = _getContainerParentsCache();
        if (cache.hasKey(coursePart.getId()))
        {
            String parentId = cache.get(coursePart.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container parentContainer = _computeParentContainerWithHigherMCCStatus(coursePart);
            cache.put(coursePart.getId(), parentContainer != null ? parentContainer.getId() : null);
            return parentContainer;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatus(CoursePart coursePart)
    {
        Set<Container> parentPrograms = _getParentYearContainers(coursePart);
        return parentPrograms.stream()
            .sorted(new MCCStatusComparator().reversed())
            .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent 'year' container with the higher MCC status from rules point of view
     * @param programItem the program item
     * @return the first parent container with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatusForRules(ProgramItem programItem)
    {
        Cache<String, String> cache = _getContainerParentsCacheForRules();
        if (cache.hasKey(programItem.getId()))
        {
            String parentId = cache.get(programItem.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container container = _computeParentContainerWithHigherMCCStatusForRules(programItem);
            cache.put(programItem.getId(), container != null ? container.getId() : null);
            return container;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatusForRules(ProgramItem programItem)
    {
        Set<Container> yearContainers = _getParentYearContainers(programItem);
        return yearContainers
                .stream()
                .sorted(new ContainerMCCRulesStatusComparator().reversed())
                .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent 'year' container with the higher MCC status from rules point of view
     * @param coursePart the course part
     * @return the first parent container with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatusForRules(CoursePart coursePart)
    {
        Cache<String, String> cache = _getContainerParentsCacheForRules();
        if (cache.hasKey(coursePart.getId()))
        {
            String parentId = cache.get(coursePart.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container container = _computeParentContainerWithHigherMCCStatusForRules(coursePart);
            cache.put(coursePart.getId(), container != null ? container.getId() : null);
            return container;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatusForRules(CoursePart coursePart)
    {
        Set<Container> yearContainers = _getParentYearContainers(coursePart);
        return yearContainers
                .stream()
                .sorted(new ContainerMCCRulesStatusComparator().reversed())
                .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent 'year' container with the higher MCC status from rules point of view
     * @param programItem the program item
     * @return the first parent container with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatusForMCCFields(ProgramItem programItem)
    {
        Cache<String, String> cache = _getContainerParentsCacheForMCC();
        if (cache.hasKey(programItem.getId()))
        {
            String parentId = cache.get(programItem.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container container = _computeParentContainerWithHigherMCCStatusForMCCFields(programItem);
            cache.put(programItem.getId(), container != null ? container.getId() : null);
            return container;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatusForMCCFields(ProgramItem programItem)
    {
        Set<Container> yearContainers = _getParentYearContainers(programItem);
        return yearContainers
                .stream()
                .sorted(new ContainerMCCFieldsStatusComparator().reversed())
                .findFirst().orElse(null);
    }
    
    /**
     * Get the first parent 'year' container with the higher MCC status from rules point of view
     * @param coursePart the course part
     * @return the first parent container with the higher MCC status or null if not found
     */
    public Container getParentContainerWithHigherMCCStatusForMCCFields(CoursePart coursePart)
    {
        Cache<String, String> cache = _getContainerParentsCacheForMCC();
        if (cache.hasKey(coursePart.getId()))
        {
            String parentId = cache.get(coursePart.getId());
            return parentId != null ? _resolver.resolveById(parentId) : null;
        }
        else
        {
            Container container = _computeParentContainerWithHigherMCCStatusForMCCFields(coursePart);
            cache.put(coursePart.getId(), container != null ? container.getId() : null);
            return container;
        }
    }
    
    private Container _computeParentContainerWithHigherMCCStatusForMCCFields(CoursePart coursePart)
    {
        Set<Container> yearContainers = _getParentYearContainers(coursePart);
        return yearContainers
                .stream()
                .sorted(new ContainerMCCFieldsStatusComparator().reversed())
                .findFirst().orElse(null);
    }
    
    /**
     * Validate MCC to orgunit level
     * @param container the container
     * @param date the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the program has changed
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate MCC to orgunit level
     */
    public boolean validateOrgunitMCC(Container container, LocalDate date, UserIdentity user, String comment)
    {
        if (!_pilotageHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (RulesManager.isRulesEnabled() && !isRulesValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be validated to orgunit level as rules are not validated for container " + container, ExceptionType.RULES_VALIDATION_REQUIRED);
        }
        
        if (!isMCCValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be validated to orgunit level as MCC are not validated for container " + container, ExceptionType.MCC_VALIDATION_REQUIRED);
        }
        
        if (isMCCOrgunitValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be validated to orgunit level as there are already validated to orgunit level for container " + container, ExceptionType.MCC_ORGUNIT_VALIDATED);
        }
        
        if (!_contentWorkflowHelper.isAvailableAction(container, MCC_WORKFLOW_ACTION_ID))
        {
            throw new MCCWorkflowException("Container data or structure are not valid. Orgunit MCC can't be validated for container : " + container, ExceptionType.WORKFLOW_INVALID);
        }
        
        return _setMCCStatus(container, __MCC_ORGUNIT_VALIDATION_PREFIX, date, user, comment);
    }
    
    /**
     * Invalidate MCC to orgunit level
     * @param container the container
     * @return <code>true</code> if the program has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to invalidate MCC for orgunit level
     */
    public boolean invalidateOrgunitMCC(Container container) throws MCCWorkflowException
    {
        if (!_pilotageHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (isMCCCFVUValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be invalidate to orgunit level as MCC are already CFVU validated for container " + container, ExceptionType.MCC_CFVU_VALIDATED);
        }
        return _removeMCCStatus(container, __MCC_ORGUNIT_VALIDATION_PREFIX);
    }
    
    /**
     * Determines if MCC are validated to orgunit level
     * @param container the container
     * @return <code>true</code> if MCC are validated to orgunit level
     */
    public boolean isMCCOrgunitValidated(Container container)
    {
        return _getMCCStatus(container, __MCC_ORGUNIT_VALIDATION_PREFIX);
    }
    
    /**
     * Validate MCC to mention level
     * @param container the container
     * @param date the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the program has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate MCC
     */
    public boolean validateMCC(Container container, LocalDate date, UserIdentity user, String comment)
    {
        if (!_pilotageHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!_contentWorkflowHelper.isAvailableAction(container, MCC_WORKFLOW_ACTION_ID))
        {
            throw new MCCWorkflowException("Container data or structure are not valid. MCC can't be validated for container : " + container, ExceptionType.WORKFLOW_INVALID);
        }
        
        return _setMCCStatus(container, __MCC_VALIDATION_PREFIX, date, user, comment);
    }
    
    /**
     * Invalidate MCC to mention level
     * @param container the container
     * @return <code>true</code> if the container has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to invalidate MCC
     */
    public boolean invalidateMCC(Container container) throws MCCWorkflowException
    {
        if (!_pilotageHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (isMCCOrgunitValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be invalidate as MCC are already validated to orgunit level for container " + container, ExceptionType.MCC_ORGUNIT_VALIDATED);
        }
        
        return _removeMCCStatus(container, __MCC_VALIDATION_PREFIX);
    }
    
    /**
     * Determines if MCC are validated to mention level
     * @param container the container
     * @return <code>true</code> if MCC are validated to mention level
     */
    public boolean isMCCValidated(Container container)
    {
        return _getMCCStatus(container, __MCC_VALIDATION_PREFIX);
    }
    
    /**
     * Validate rules
     * @param container the container
     * @param date the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the program has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate rules
     */
    public boolean validateRules(Container container, LocalDate date, UserIdentity user, String comment)
    {
        if (!_pilotageHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (isMCCOrgunitValidated(container))
        {
            throw new MCCWorkflowException("Rules can not be invalidate as rules are already validated to orgunit level for container " + container, ExceptionType.MCC_ORGUNIT_VALIDATED);
        }
        
        if (!_isRuleActionAvailable(container))
        {
            throw new MCCWorkflowException("Container data or structure are not valid. Rules can't be validated for container : " + container, ExceptionType.WORKFLOW_INVALID);
        }
        
        return _setMCCStatus(container, __RULES_VALIDATION_PREFIX, date, user, comment);
    }
    
    /**
     * Invalidate rules to mention level
     * @param container the container
     * @return <code>true</code> if the container has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to invalidate rules
     */
    public boolean invalidateRules(Container container) throws MCCWorkflowException
    {
        if (!_pilotageHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (isMCCOrgunitValidated(container))
        {
            throw new MCCWorkflowException("Rules can not be invalidate as rules are already validated to orgunit level for container " + container, ExceptionType.MCC_ORGUNIT_VALIDATED);
        }
        
        return _removeMCCStatus(container, __RULES_VALIDATION_PREFIX);
    }
    
    /**
     * Determines if rules are validated to mention level
     * @param container the container
     * @return <code>true</code> if rules are validated to mention level
     */
    public boolean isRulesValidated(Container container)
    {
        return _getMCCStatus(container, __RULES_VALIDATION_PREFIX);
    }
    
    /**
     * Determines if MCC are validated
     * @param container the container
     * @return <code>true</code> if MCC are validated
     */
    public boolean isMCCCFVUValidated(Container container)
    {
        return _getMCCStatus(container, __CFVU_MCC_VALIDATION_PREFIX);
    }
    
    /**
     * Determines if rules can be validated
     * @param container the container
     * @return <code>true</code> if rules can be validated
     */
    public boolean canValidateRules(Container container)
    {
        return _isRuleActionAvailable(container);
    }
    
    private boolean _isRuleActionAvailable(Container container)
    {
        // For rule validation, we don't want to check the container structure. So we put CHECK_TREE_KEY to false
        Map<String, Object> inputs = new HashMap<>();
        inputs.put(ValidateProgramItemTreeCondition.CHECK_TREE_KEY, false);
        return _contentWorkflowHelper.isAvailableAction(container, MCC_WORKFLOW_ACTION_ID, inputs);
    }
    
    /**
     * Determines if MCC can be validated
     * @param container the container
     * @return <code>true</code> if MCC can be validated
     */
    public boolean canValidateMCC(Container container)
    {
        return _contentWorkflowHelper.isAvailableAction(container, MCC_WORKFLOW_ACTION_ID);
    }
    
    /**
     * Determines if MCC can be validated to orgunit level
     * @param container the container
     * @return <code>true</code> if MCC can be validated to orgunit level
     */
    public boolean canValidateOrgUnitMCC(Container container)
    {
        return isMCCValidated(container) && (!RulesManager.isRulesEnabled() || isRulesValidated(container)) && _contentWorkflowHelper.isAvailableAction(container, MCC_WORKFLOW_ACTION_ID);
    }
    
    /**
     * Determines if MCC can be CFVU validated
     * @param container the container
     * @return <code>true</code> if MCC can be CFVU validated
     */
    public boolean canValidateCFVUMCC(Container container)
    {
        return isMCCOrgunitValidated(container) && _contentWorkflowHelper.isAvailableAction(container, MCC_WORKFLOW_ACTION_ID);
    }
    
    
    /**
     * Get minimun date for MCC orgunit validation
     * @param container the container
     * @return the minimun date
     */
    public LocalDate getMinDateForMCCOrgUnitValidation(Container container)
    {
        if (!isMCCValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be validated as MCC are not validated to mention level for container " + container, ExceptionType.MCC_VALIDATION_REQUIRED);
        }
        
        if (RulesManager.isRulesEnabled() && !isRulesValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be validated as rules are not validated to mention level for container " + container, ExceptionType.RULES_VALIDATION_REQUIRED);
        }
        
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
        LocalDate mccMentionDate = composite.getValue(__MCC_VALIDATION_PREFIX + __DATE_SUFFIX);
        LocalDate rulesMentionDate = RulesManager.isRulesEnabled() ? composite.getValue(__RULES_VALIDATION_PREFIX + __DATE_SUFFIX) : null;
        
        return rulesMentionDate == null ? mccMentionDate : mccMentionDate.compareTo(rulesMentionDate) > 0 ? mccMentionDate : rulesMentionDate;
    }
    
    /**
     * Get minimun date for MCC CFVU validation
     * @param container the container
     * @return the minimun date
     */
    public LocalDate getMinDateForMCCCFVUValidation(Container container)
    {
        if (!isMCCOrgunitValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be CFVU validated as MCC are not validated to orgunit level for container " + container, ExceptionType.MCC_ORGUNIT_VALIDATION_REQUIRED);
        }
        
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
        return composite.getValue(__MCC_ORGUNIT_VALIDATION_PREFIX + __DATE_SUFFIX);
    }
    
    /**
     * Get the MCC CFVU validation date
     * @param container the container
     * @return the MCC CFVU validation date
     */
    public LocalDate getMCCCFVUValidationDate(Container container)
    {
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
        return composite.getValue(__CFVU_MCC_VALIDATION_PREFIX + __DATE_SUFFIX);
    }
    
    /**
     * Validate MCC for this program
     * @param container the container
     * @param validationDate the validation date
     * @param user the user
     * @param comment the comment
     * @return <code>true</code> if the container has changed.
     * @throws MCCWorkflowException if the MCC workflow does not allow to validate MCC to orgunit level
     */
    public boolean validateMCCForCVFU(Container container, LocalDate validationDate, UserIdentity user, String comment)
    {
        if (!_pilotageHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        if (!isMCCOrgunitValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be CFVU validated as MCC are not validated to orgunit level for container " + container, ExceptionType.MCC_ORGUNIT_VALIDATION_REQUIRED);
        }
        
        if (isMCCCFVUValidated(container))
        {
            throw new MCCWorkflowException("MCC can not be CFVU validated as MCC are already validated to CFVU for container " + container, ExceptionType.MCC_CFVU_VALIDATED);
        }
        
        if (!_contentWorkflowHelper.isAvailableAction(container, MCC_WORKFLOW_ACTION_ID))
        {
            throw new MCCWorkflowException("Container data or structure are not valid. MCC for CFVU can't be validated for container : " + container, ExceptionType.WORKFLOW_INVALID);
        }
        
        return _setMCCStatus(container, __CFVU_MCC_VALIDATION_PREFIX, validationDate, user, comment);
    }
    
    /**
     * Remove validation attribute for 'CFVU MCC validated' state
     * @param container the container
     * @return <code>true</code> if the container has changed.
     */
    public boolean invalidateMCCForCVFU(Container container)
    {
        if (!_pilotageHelper.isContainerOfTypeYear(container))
        {
            throw new MCCWorkflowException("Can not update MCC status on a container that is not of type year: " + container, ExceptionType.NOT_TYPE_YEAR);
        }
        
        return _removeMCCStatus(container, __CFVU_MCC_VALIDATION_PREFIX);
    }
    
    /**
     * Removes attributes linked to MCC workflow
     * @param container the container
     * @return <code>true</code> if the program has changed.
     */
    public boolean removeMCCWorkflow(Container container)
    {
        container.removeValue(__MCC_WORKFLOW_COMPOSITE);
        return saveContent(container);
    }
    
    /**
     * Removes attributes linked to MCC workflow
     * @param container the container
     * @return <code>true</code> if the program has changed.
     */
    public boolean removeMCCValidatedPDF(Container container)
    {
        container.removeValue(MCC_VALIDATED_PDF_REPEATER);
        return saveContent(container);
    }
    
    private boolean _setMCCStatus(Container container, String prefixAttributeName, LocalDate validationDate, UserIdentity user, String comment)
    {
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE, true);
        composite.setValue(prefixAttributeName + __STATUS_SUFFIX, true);
        setWorkflowStep(container, composite, prefixAttributeName, validationDate, user, comment);
        
        return saveContentAndNotify(container);
    }
    
    private boolean _removeMCCStatus(Container container, String prefixAttributeName)
    {
        ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE, true);
        composite.setValue(prefixAttributeName + __STATUS_SUFFIX, false);
        removeWorkflowStep(container, composite, prefixAttributeName);
        
        return saveContentAndNotify(container);
    }
    
    private boolean _getMCCStatus(Container container, String attributeName)
    {
        return container.getValue(__MCC_WORKFLOW_COMPOSITE + ModelItem.ITEM_PATH_SEPARATOR + attributeName + __STATUS_SUFFIX, false, false);
    }
    
    /**
     * Get the MCC workflow step for validation of rules to mention level
     * @param container the container
     * @return the MCC workflow step or null if rules are not validated to mention level
     */
    public ODFWorkflowStep getRulesMentionValidationStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.RULES_VALIDATED, __RULES_VALIDATION_PREFIX);
    }
    
    /**
     * Get the MCC workflow step for validation of MCC to mention level
     * @param container the container
     * @return the MCC workflow step or null if MCC are not validated to mention level
     */
    public ODFWorkflowStep getMCCMentionValidationStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.MCC_VALIDATED, __MCC_VALIDATION_PREFIX);
    }

    /**
     * Get the MCC workflow step for validation of MCC to orgunit level
     * @param container the container
     * @return the MCC workflow step or null if MCC are not validated to orgunit level
     */
    public ODFWorkflowStep getMCCOrgunitValidationStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.MCC_ORGUNIT_VALIDATED, __MCC_ORGUNIT_VALIDATION_PREFIX);
    }
    
    /**
     * Get the MCC workflow step for CFVU validation of MCC
     * @param container the container
     * @return the MCC workflow step or null if MCC are not CFVU validated
     */
    public ODFWorkflowStep getCFVUMCCValidationStep(Container container)
    {
        return _getMCCWorkflowStep(container, MCCWorkflowStep.CFVU_MCC_VALIDATED, __CFVU_MCC_VALIDATION_PREFIX);
    }
    
    private ODFWorkflowStep _getMCCWorkflowStep(Container container, MCCWorkflowStep step, String attributeNamePrefix)
    {
        if (container.hasValue(__MCC_WORKFLOW_COMPOSITE))
        {
            boolean status = _getMCCStatus(container, attributeNamePrefix);
            if (status)
            {
                ModifiableModelAwareComposite composite = container.getComposite(__MCC_WORKFLOW_COMPOSITE);
                return getWorkflowStep(composite, attributeNamePrefix, step.name());
            }
        }
        
        return null;
    }
    
    /**
     * Get the entry holding the most recent PDF for validated MCC on a year.
     * @param container the container
     * @return the entry or null if no PDF available
     */
    public ModelAwareRepeaterEntry getLastMCCValidatedEntry(Container container)
    {
        return Optional.ofNullable(container.getRepeater(MCC_VALIDATED_PDF_REPEATER))
            .map(ModelAwareRepeater::getEntries)
            .map(List::stream)
            .orElseGet(Stream::of)
            .filter(e -> e.hasValue("pdf"))
            .max((entry1, entry2) -> entry1.<LocalDate>getValue("date").compareTo(entry2.<LocalDate>getValue("date")))
            .orElse(null);
    }
    
    private final class MCCStatusComparator implements Comparator<Container>
    {
        public int compare(Container c1, Container c2)
        {
            boolean mccValidated1 = isMCCCFVUValidated(c1);
            boolean mccValidated2 = isMCCCFVUValidated(c2);
            
            if (mccValidated1)
            {
                return mccValidated2 ? 0 : 1;
            }
            
            if (mccValidated2)
            {
                return -1; // p2 has a higher mcc status
            }
            
            boolean mccOrgunitValidated1 = isMCCOrgunitValidated(c1);
            boolean mccOrgunitValidated2 = isMCCOrgunitValidated(c2);
            
            if (mccOrgunitValidated1)
            {
                return mccOrgunitValidated2 ? 0 : 1;
            }
            if (mccOrgunitValidated2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean rulesMentionValidated1 = RulesManager.isRulesEnabled() && isRulesValidated(c1);
            boolean mccMentionValidated1 = isMCCValidated(c1);
            boolean rulesValidated2 = RulesManager.isRulesEnabled() && isRulesValidated(c2);
            boolean mccMentionValidated2 = isMCCValidated(c2);
            
            if (rulesMentionValidated1 && mccMentionValidated1)
            {
                if (!mccMentionValidated2 || !rulesValidated2)
                {
                    return 1; // c1 has a higher mcc status
                }
                else
                {
                    return 0; // same status
                }
            }
            
            if (rulesMentionValidated1 || mccMentionValidated1)
            {
                if (rulesValidated2 && !mccMentionValidated2 || !rulesValidated2 && mccMentionValidated2)
                {
                    return 0; // same status
                }
                else if (rulesValidated2 && mccMentionValidated2)
                {
                    return -1; // c2 has a higher mcc status
                }
                else
                {
                    return 1; // c1 has a higher mcc status
                }
            }
            
            return !rulesValidated2 && !mccMentionValidated2 ? 0 : -1;
        }
    }
    
    private final class ContainerMCCRulesStatusComparator implements Comparator<Container>
    {
        public int compare(Container c1, Container c2)
        {
            boolean mccValidated1 = isMCCCFVUValidated(c1);
            boolean mccValidated2 = isMCCCFVUValidated(c2);
            
            if (mccValidated1)
            {
                return mccValidated2 ? 0 : 1;
            }
            
            if (mccValidated2)
            {
                return -1; // p2 has a higher mcc status
            }
            
            boolean mccOrgunitValidated1 = isMCCOrgunitValidated(c1);
            boolean mccOrgunitValidated2 = isMCCOrgunitValidated(c2);
            
            if (mccOrgunitValidated1)
            {
                return mccOrgunitValidated2 ? 0 : 1;
            }
            if (mccOrgunitValidated2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean rulesValidated1 = RulesManager.isRulesEnabled() && isRulesValidated(c1);
            boolean rulesValidated2 = RulesManager.isRulesEnabled() && isRulesValidated(c2);
            
            if (rulesValidated1)
            {
                return rulesValidated2 ? 0 : 1;
            }
            
            return rulesValidated2 ? -1 : 0;
        }
    }
    
    private final class ContainerMCCFieldsStatusComparator implements Comparator<Container>
    {
        public int compare(Container c1, Container c2)
        {
            boolean mccValidated1 = isMCCCFVUValidated(c1);
            boolean mccValidated2 = isMCCCFVUValidated(c2);
            
            if (mccValidated1)
            {
                return mccValidated2 ? 0 : 1;
            }
            
            if (mccValidated2)
            {
                return -1; // p2 has a higher mcc status
            }
            
            boolean mccOrgunitValidated1 = isMCCOrgunitValidated(c1);
            boolean mccOrgunitValidated2 = isMCCOrgunitValidated(c2);
            
            if (mccOrgunitValidated1)
            {
                return mccOrgunitValidated2 ? 0 : 1;
            }
            if (mccOrgunitValidated2)
            {
                return -1; // c2 has a higher mcc status
            }
            
            boolean mccMentionValidated1 = isMCCValidated(c1);
            boolean mccMentionValidated2 = isMCCValidated(c2);
            
            if (mccMentionValidated1)
            {
                return mccMentionValidated2 ? 0 : 1;
            }
            
            return mccMentionValidated2 ? -1 : 0;
        }
    }
}
