/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.property;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.model.properties.AbstractProperty;
import org.ametys.cms.model.properties.Property;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowStep;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * {@link Property} for MCC workflow status on a {@link Program}
 */
public class ContainerMCCWorkflowStatusProperty extends AbstractProperty<String, Container> implements Serviceable
{
    private MCCWorkflowHelper _mccWorkflowHelper;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _mccWorkflowHelper = (MCCWorkflowHelper) smanager.lookup(MCCWorkflowHelper.ROLE);
    }
    
    public Object getValue(Container container)
    {
        List<String> status = new ArrayList<>();
        if (_mccWorkflowHelper.isMCCCFVUValidated(container))
        {
            status.add(MCCWorkflowStep.CFVU_MCC_VALIDATED.name());
        }
        else if (_mccWorkflowHelper.isMCCOrgunitValidated(container))
        {
            status.add(MCCWorkflowStep.MCC_ORGUNIT_VALIDATED.name());
        }
        else 
        {
            if (_mccWorkflowHelper.isRulesValidated(container))
            {
                status.add(MCCWorkflowStep.RULES_VALIDATED.name());
            }
            
            if (_mccWorkflowHelper.isMCCValidated(container))
            {
                status.add(MCCWorkflowStep.MCC_VALIDATED.name());
            }
        }
        return status.toArray(String[]::new);
    }
    
    @Override
    public Enumerator<String> getEnumerator()
    {
        StaticEnumerator<String> enumerator = new StaticEnumerator<>();
        enumerator.addAll(Map.of(
                MCCWorkflowStep.CFVU_MCC_VALIDATED.name(), new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_CFVU_VALIDATED_LABEL"),
                MCCWorkflowStep.MCC_ORGUNIT_VALIDATED.name(), new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_ORGUNIT_VALIDATED_LABEL"),
                MCCWorkflowStep.RULES_VALIDATED.name(), new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_RULES_VALIDATED_LABEL"),
                MCCWorkflowStep.MCC_VALIDATED.name(), new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_VALIDATED_LABEL")
        ));
        
        return enumerator;
    }

    @Override
    public boolean isMultiple()
    {
        return true;
    }
    
    @Override
    protected String _getTypeId()
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
    
}
