/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.consistency.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper.StepHolderStatus;
import org.ametys.plugins.odfpilotage.report.consistency.AbstractConsistencyAnalysis;
import org.ametys.plugins.odfpilotage.report.consistency.ConsistencyAnalysisResult;
import org.ametys.plugins.odfpilotage.report.consistency.ConsistencyAnalysisStatus;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Analysis on step holder for all courses of the {@link Program}.
 */
public class StepHolderAnalysis extends AbstractConsistencyAnalysis
{
    /** The pilotage helper */
    protected PilotageHelper _pilotageHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
    }
    
    
    @Override
    public ConsistencyAnalysisResult analyze(ProgramItem programItem)
    {
        ConsistencyAnalysisResult result = new ConsistencyAnalysisResult();

        // Introduction
        result.setIntroText(new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_STEPHOLDER_INTRO"));
        
        // Columns
        result.addColumn("courseCode", new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_STEPHOLDER_COLUMN_COURSE_CODE"));
        result.addColumn("courseTitle", new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_STEPHOLDER_COLUMN_COURSE_TITLE"));
        result.addColumn("stepHolderStatus", new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_STEPHOLDER_COLUMN_STEP_HOLDER_STATUS"));
        
        // Errors
        List<Map<String, Object>> errors = _controlStepHolderOnCourses(programItem);
        result.addLines(errors);
        
        // Status
        result.setStatus(errors.isEmpty() ? ConsistencyAnalysisStatus.OK : ConsistencyAnalysisStatus.KO);
        
        return result;
    }
    
    /**
     * Control the ECTS consistency between the current content and its children.
     * @param programItem The {@link Program} element to check step holder on each course
     * @return The {@link List} of error messages
     */
    protected List<Map<String, Object>> _controlStepHolderOnCourses(ProgramItem programItem)
    {
        List<Map<String, Object>> errors = new ArrayList<>();

        // For each course of the program, check if the step holder is single, if not, add a line to the report
        for (Course course : getAllCourses(programItem))
        {
            StepHolderStatus stepHolderStatus = _pilotageHelper.getStepHolder(course).getLeft();
            if (stepHolderStatus != StepHolderStatus.SINGLE)
            {
                Map<String, Object> error = new HashMap<>();
                error.put("courseCode", course.getCode());
                error.put("courseTitle", course.getTitle());
                error.put("stepHolderStatus", new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_STEPHOLDER_STATUS_" + stepHolderStatus.toString()));
                errors.add(error);
            }
        }
        
        return errors;
    }
    
    private Set<Course> getAllCourses(ProgramItem programItem)
    {
        Set<Course> courses = new HashSet<>();
        
        boolean alreadyAdded = false;
        if (programItem instanceof Course course)
        {
            alreadyAdded = !courses.add(course);
        }
        
        if (!alreadyAdded)
        {
            _odfHelper.getChildProgramItems(programItem).forEach(child -> courses.addAll(getAllCourses(child)));
        }
        
        return courses;
    }
}
