/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.holder.DataHolderDisableConditionsEvaluator;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.odfpilotage.report.impl.mcc.MCCProgramItemTree;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.disableconditions.DisableConditionsEvaluator;

/**
 * Class to generate a report based on MCC.
 */
public abstract class AbstractMCCReport extends AbstractReport implements Initializable
{
    /** The key to define if we want a result by program or year */
    public static final String PARAMETER_ONE_FILE_BY_PROGRAM = "oneFileByProgram";
    
    /** Prefix of the sessions name */
    private static final String __SESSION_NAME_PREFIX = "mccSession";
    
    /** Name of the first session */
    protected static final String FIRST_SESSION_NAME = __SESSION_NAME_PREFIX + "1";
    /** Name of the second session */
    protected static final String SECOND_SESSION_NAME = __SESSION_NAME_PREFIX + "2";
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    /** The disable conditions evaluator */
    protected DisableConditionsEvaluator<ModelAwareDataHolder> _disableConditionsEvaluator;
    
    /** The content type EP */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    /** The MCC sessions view */
    protected View _sessionsView;
    
    private View _programView;
    private View _subProgramView;
    private View _containerView;
    private View _courseListView;
    private View _courseView;
    
    @Override
    @SuppressWarnings("unchecked")
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _disableConditionsEvaluator = (DisableConditionsEvaluator<ModelAwareDataHolder>) manager.lookup(DataHolderDisableConditionsEvaluator.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _programView = _getView(ProgramFactory.PROGRAM_CONTENT_TYPE, "report-mcc");
        _subProgramView = _getView(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, "report-mcc");
        _containerView = _getView(ContainerFactory.CONTAINER_CONTENT_TYPE, "report-mcc");
        _courseListView = _getView(CourseListFactory.COURSE_LIST_CONTENT_TYPE, "report-mcc");
        _courseView = _getView(CourseFactory.COURSE_CONTENT_TYPE, "report-mcc");
        _sessionsView = _getView(CourseFactory.COURSE_CONTENT_TYPE, "report-mcc-sessions");
    }
    
    @Override
    protected boolean isGeneric()
    {
        return false;
    }
    
    @Override
    public String getDefaultOutputFormat()
    {
        return OUTPUT_FORMAT_PDF;
    }
    
    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_DOC, OUTPUT_FORMAT_XLS, OUTPUT_FORMAT_PDF);
    }
    
    private View _getView(String contentTypeId, String viewName)
    {
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        return contentType.getView(viewName);
    }
    
    @Override
    protected PilotageReportContent getReportContentForOrgUnit(String outputFormat, Map<String, String> reportParameters)
    {
        return _getReportContentForProgramItemsInOrgUnit(outputFormat, reportParameters);
    }
    
    @Override
    protected Stream<? extends ProgramItem> _getProgramItemsFromProgram(Program program, Map<String, String> reportParameters)
    {
        if (!Boolean.parseBoolean(reportParameters.get(PARAMETER_ONE_FILE_BY_PROGRAM)))
        {
            return _pilotageHelper.getYears(program).stream();
        }
        return super._getProgramItemsFromProgram(program, reportParameters);
    }
    
    @Override
    public void saxProgramItem(ContentHandler handler, String programItemId, Map<String, String> reportParameters)
    {
        // The MCC (diff too) report can accept every program item
        ProgramItem programItem = _resolver.resolveById(programItemId);
        
        MCCProgramItemTree programTree = _createMCCTreeFromProgramItem(programItem, reportParameters);
        
        try
        {
            handler.startDocument();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType());
            XMLUtils.startElement(handler, "report", attrs);

            // SAX other global informations
            saxGlobalInformations(handler, programItem, reportParameters);
            
            // SAX natures enseignement
            _reportHelper.saxNaturesEnseignement(handler, getLogger());
            
            // SAX tree
            saxTree(handler, programTree);
            
            XMLUtils.endElement(handler, "report");
            
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'MCC' report for program item '{}' ({})", ((Content) programItem).getTitle(), programItem.getCode(), e);
        }
    }

    @Override
    public void _saxOrgUnit(ContentHandler handler, String catalog, String lang, String orgUnitId, Map<String, String> reportParameters)
    {
        throw new UnsupportedOperationException("Impossible to launch the report '" + getType() + "' on a orgunit.");
    }
    
    /**
     * SAX the MCC tree for the report
     * @param handler The handler
     * @param tree The MCC tree
     * @throws SAXException if an error occurs
     */
    protected void saxTree(ContentHandler handler, MCCProgramItemTree tree) throws SAXException
    {
        ProgramItem programItem = tree.getCurrent();
        
        if (programItem instanceof Program program)
        {
            _saxContent(handler, "program", program, _programView, tree);
        }
        else if (programItem instanceof SubProgram subProgram)
        {
            _saxContent(handler, "subProgram", subProgram, _subProgramView, tree);
        }
        else if (programItem instanceof Container container)
        {
            _saxContent(handler, "container", container, _containerView, tree);
        }
        else if (programItem instanceof CourseList courseList)
        {
            _saxContent(handler, "courseList", courseList, _courseListView, tree);
        }
        else if (programItem instanceof Course course)
        {
            _saxContent(handler, "course", course, _courseView, tree);
        }
    }
    
    /**
     * Sax a content on a view to the handler and sax children.
     * @param handler The handler
     * @param tagName The root tag name
     * @param content The content
     * @param view The view
     * @param tree The tree for children
     * @throws SAXException if an error occurs
     */
    protected void _saxContent(ContentHandler handler, String tagName, Content content, View view, MCCProgramItemTree tree) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", content.getId());
        attrs.addCDATAAttribute("name", content.getName());
        attrs.addCDATAAttribute("path", tree.getPath().toString());
        if (_pilotageHelper.canWriteMccRestrictions(content, "mccSession2"))
        {
            attrs.addCDATAAttribute("session2", "available");
        }
        XMLUtils.startElement(handler, tagName, attrs);
        
        // Sax content data
        content.dataToSAX(handler, view);
        
        // Sax MCC for courses
        if (content instanceof Course course)
        {
            _saxMCCs(handler, course, tree);
        }
        
        // Sax children
        for (MCCProgramItemTree child : tree.getChildren())
        {
            saxTree(handler, child);
        }
        
        XMLUtils.endElement(handler, tagName);
    }
    
    /**
     * Sax the MCC sessions.
     * @param handler The handler
     * @param course The concerned {@link Course} to sax the MCCs on
     * @param tree The MCC tree
     * @throws SAXException If an error occurs
     */
    protected abstract void _saxMCCs(ContentHandler handler, Course course, MCCProgramItemTree tree) throws SAXException;
    
    /**
     * Generates SAX events for the details of a MCC session entry.
     * @param handler The handler
     * @param sessionEntry The session repeater entry to SAX
     * @param rootContent The root content of the session
     * @throws SAXException If an error occurs
     */
    protected void _saxSessionEntryDetails(ContentHandler handler, ModelAwareRepeaterEntry sessionEntry, Content rootContent) throws SAXException
    {
        String repeaterName = sessionEntry.getHoldingRepeater().getModel().getName();
        sessionEntry.dataToSAX(
            handler,
            (ViewItemAccessor) _sessionsView.getModelViewItem(repeaterName),
            RepositoryDataContext.newInstance()
                .withObject(rootContent)
                .withDataPath(repeaterName + "[" + sessionEntry.getPosition() + "]")
        );
    }

    /**
     * Add and populate the program item to the {@link MCCProgramItemTree}
     * @param programItem The program item to add
     * @param reportParameters The report parameters
     * @return a tree of the program
     */
    protected MCCProgramItemTree _createMCCTreeFromProgramItem(ProgramItem programItem, Map<String, String> reportParameters)
    {
        MCCProgramItemTree programItemTree = new MCCProgramItemTree(programItem);
        populateMCCTree(programItemTree);
        return programItemTree;
    }
    
    /**
     * Populate the MCC tree.
     * @param tree The MCC tree
     */
    protected abstract void populateMCCTree(MCCProgramItemTree tree);
    
    /**
     * Sax the additional global informations of the report.
     * @param handler The handler
     * @param programItem The program item on which the report is launched
     * @param reportParameters The report parameters
     * @throws SAXException If an error occurs
     */
    protected void saxGlobalInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        String catalog = programItem.getCatalog();
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", catalog);
        XMLUtils.createElement(handler, "catalog", attrs, _catalogsManager.getCatalog(catalog).getTitle());
        XMLUtils.createElement(handler, "title", ((Content) programItem).getTitle());
        XMLUtils.createElement(handler, "lang", programItem.getLanguage());
        for (EducationalPath path : _odfHelper.getEducationalPaths(programItem, false, true))
        {
            String readablePath = path.resolveProgramItems(_resolver).map(Content.class::cast).map(Content::getTitle).collect(Collectors.joining(" > "));
            XMLUtils.createElement(handler, "path", readablePath);
        }
    }
}
