/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.observations;

import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.commons.collections4.ListUtils;
import org.apache.commons.collections4.SetUtils;
import org.apache.commons.collections4.SetUtils.SetView;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.TraversableProgramPart;

/**
 * Observer to remove derogation and additional rule if the structure of the traversable program part degree is modified.
 */
public class UpdateTraversableProgramPartRulesStep2Observer extends UpdateProgramRulesStep2Observer
{
    @Override
    protected boolean supportsContent(Content content)
    {
        return content instanceof TraversableProgramPart;
    }
    
    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        TraversableProgramPart traversableProgramPart = (TraversableProgramPart) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        
        Set<Container> newChildContainers = _getChildContainerOfTypeYear(traversableProgramPart);
        Set<Container> oldChildContainers = _getOldChildContainers((Content) traversableProgramPart);
        
        // Only process if the child containers has been modified
        if (!ListUtils.isEqualList(newChildContainers, oldChildContainers))
        {
            SetView<Container> commonContainers = SetUtils.intersection(newChildContainers, oldChildContainers);
            Set<Container> incompatibleContainers = newChildContainers.stream()
                    .filter(Predicate.not(commonContainers::contains))
                    .collect(Collectors.toSet());
            
            incompatibleContainers.addAll(oldChildContainers.stream()
                    .filter(Predicate.not(commonContainers::contains))
                    .collect(Collectors.toSet())
            );
            
            for (Container container : incompatibleContainers)
            {
                _deleteOldRulesOnContainer(container);
            }
        }
    }
}
