/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.tool;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.comparator.LastModifiedFileComparator;
import org.apache.commons.io.comparator.NameFileComparator;
import org.apache.commons.io.comparator.SizeFileComparator;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.languages.Language;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.ServerCommHelper;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.helper.ReportHelper;
import org.ametys.plugins.odfpilotage.manager.PilotageLogFileManager;
import org.ametys.plugins.odfpilotage.report.AbstractPilotageReport;
import org.ametys.plugins.odfpilotage.report.PilotageReport;
import org.ametys.plugins.odfpilotage.report.PilotageReport.PilotageReportTarget;
import org.ametys.plugins.odfpilotage.report.ReportExtensionPoint;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.OrgUnitReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.ProgramReportSchedulable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

import com.google.common.collect.Lists;

/**
 * SAX the last 30 log files
 *
 */
public class ListReportsAction extends ServiceableAction
{
    private static final String _CRITERIA_FILENAME = "filename";
    private static final String _CRITERIA_LAST_MODIFIED_AFTER = "lastModifiedAfter";
    private static final String _CRITERIA_LAST_MODIFIED_BEFORE = "lastModifiedBefore";

    private static final String _COLUMN_FILENAME = "reportfile";
    private static final String _COLUMN_LAST_MODIFIED = "lastModified";
    private static final String _COLUMN_LENGTH = "length";
    private static final String _COLUMN_TYPE = "type";
    private static final String _COLUMN_GENERATION_DATE = "generationDate";
    private static final String _COLUMN_OUTPUT_FORMAT = "outputFormat";
    private static final String _COLUMN_CATALOG = "catalog";
    private static final String _COLUMN_LANG = "lang";
    private static final String _COLUMN_TARGET = "target";
    private static final String _COLUMN_CONTEXT = "context";
    private static final String _COLUMN_MANIFEST = "manifest";
    private static final String _COLUMN_LOG_FILE = "logfile";
    
    private static final Map<String, Comparator<File>> _NAME_TO_COMPARATOR = new HashMap<>();
    static
    {
        _NAME_TO_COMPARATOR.put(_COLUMN_FILENAME, NameFileComparator.NAME_INSENSITIVE_COMPARATOR);
        _NAME_TO_COMPARATOR.put(_COLUMN_LAST_MODIFIED, LastModifiedFileComparator.LASTMODIFIED_COMPARATOR);
        _NAME_TO_COMPARATOR.put(_COLUMN_LENGTH, SizeFileComparator.SIZE_COMPARATOR);
    }
    
    /** ServerComm Helper */
    protected ServerCommHelper _serverCommHelper;
    /** JSON Utils */
    protected JSONUtils _jsonUtils;
    /** Pilotage log file manager */
    protected PilotageLogFileManager _pilotageLogFileManager;
    /** Pilotage helper */
    protected ReportHelper _reportHelper;
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The report extension point */
    protected ReportExtensionPoint _reportEP;
    /** The language manager */
    protected LanguagesManager _languageManager;
    /** The catalog manager */
    protected CatalogsManager _catalogManager;
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _serverCommHelper = (ServerCommHelper) smanager.lookup(ServerCommHelper.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
        _pilotageLogFileManager = (PilotageLogFileManager) smanager.lookup(PilotageLogFileManager.ROLE);
        _reportHelper = (ReportHelper) smanager.lookup(ReportHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _reportEP = (ReportExtensionPoint) smanager.lookup(ReportExtensionPoint.ROLE);
        _languageManager = (LanguagesManager) smanager.lookup(LanguagesManager.ROLE);
        _catalogManager = (CatalogsManager) smanager.lookup(CatalogsManager.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
    }
    
    @SuppressWarnings("unchecked")
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        // Get JS parameters
        Map<String, Object> jsParameters = _serverCommHelper.getJsParameters();
        
        // Search
        Map<String, Object> searchParams = (Map<String, Object>) jsParameters.get("values");
        File[] reportFiles = _getReportFiles(searchParams);
        Integer offset = _getIntValue(jsParameters, "start", 0);
        Integer limit = _getIntValue(jsParameters, "limit", Integer.MAX_VALUE);
        List<Map<String, Object>> results = _getReportResults(reportFiles, offset, limit, _getSortList(jsParameters.get("sort")));

        // Construct the JSON object
        Map<String, Object> objectToRead = new HashMap<>();
        objectToRead.put("items", results);
        objectToRead.put("total", reportFiles.length);
        
        // Add JSON to the request to be parsed
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, objectToRead);
        
        return EMPTY_MAP;
    }

    private int _getIntValue(Map<String, Object> values, String key, int defaultValue)
    {
        if (values.containsKey(key))
        {
            return Integer.valueOf(values.get(key).toString()).intValue();
        }
        
        return defaultValue;
    }
    
    private File[] _getReportFiles(Map<String, Object> parameters)
    {
        String filename = null;
        ZonedDateTime lastModifiedAfter = null;
        ZonedDateTime lastModifiedBefore = null;
        if (parameters != null && !parameters.isEmpty())
        {
            filename = MapUtils.getString(parameters, _CRITERIA_FILENAME);
            lastModifiedAfter = _getZonedDateTimeFromParameters(parameters, _CRITERIA_LAST_MODIFIED_AFTER);
            lastModifiedBefore = _getZonedDateTimeFromParameters(parameters, _CRITERIA_LAST_MODIFIED_BEFORE);
        }
        FileFilter filter = new PilotageFileFilter(filename, lastModifiedAfter, lastModifiedBefore);

        return _reportHelper.getPilotageFolder().listFiles(filter);
    }
    
    private ZonedDateTime _getZonedDateTimeFromParameters(Map<String, Object> parameters, String parameterName)
    {
        String dateAsString = MapUtils.getString(parameters, parameterName);
        return StringUtils.isNotEmpty(dateAsString) ? ZonedDateTime.parse(dateAsString, DateTimeFormatter.ISO_DATE_TIME) : null;
    }
    
    private List<Map<String, Object>> _getReportResults(File[] reportFiles, Integer offset, Integer limit, List<Object> sort)
    {
        List<Map<String, Object>> reports = new LinkedList<>();
        
        int count = 0;

        for (File reportFile : _sortFiles(reportFiles, sort))
        {
            if (count >= offset && count < offset + limit)
            {
                String filename = reportFile.getName();
                
                Map<String, Object> report = new HashMap<>();
                report.put(_COLUMN_FILENAME, filename);
                report.put(_COLUMN_LENGTH, String.valueOf(reportFile.length()));
                report.put(_COLUMN_LAST_MODIFIED, DateUtils.epochMilliToString(reportFile.lastModified()));
                
                // Parse manifest.json (if it exists)
                Map<String, Object> manifestData = _getManifestData(reportFile);
                if (manifestData != null)
                {
                    report.put(_COLUMN_MANIFEST, manifestData);
                    report.putAll(_convertManifestToColumns(manifestData));

                    // Get log file from manifest data
                    File logFile = _getLogFile(manifestData);
                    if (logFile != null)
                    {
                        report.put(_COLUMN_LOG_FILE, logFile.getName());
                    }
                }
                
                reports.add(report);
            }
            else if (count >= offset + limit)
            {
                break;
            }
            count++;
        }
        
        return reports;
    }
    
    private Map<String, Object> _getManifestData(File reportFile)
    {

        try (ZipFile zipFile = new ZipFile(reportFile);)
        {
            ZipEntry zipEntry = zipFile.getEntry(AbstractPilotageReport.MANIFEST_FILENAME);
            if (zipEntry != null)
            {
                try (InputStream is = zipFile.getInputStream(zipEntry);)
                {
                    String manifestContent = IOUtils.toString(is, StandardCharsets.UTF_8);
                    return _jsonUtils.convertJsonToMap(manifestContent);
                }
            }
        }
        catch (IOException e)
        {
            getLogger().error("An error occured while reading the manifest.json file of " + reportFile.getName(), e);
        }
        
        return null;
    }
    
    private Map<String, Object> _convertManifestToColumns(Map<String, Object> manifestData)
    {
        Map<String, Object> infos = new HashMap<>();
        infos.put(_COLUMN_TYPE, _getReportTypeInfos((String) manifestData.get("type")));
        infos.put(_COLUMN_GENERATION_DATE, manifestData.get("date"));
        infos.put(_COLUMN_OUTPUT_FORMAT, manifestData.get(AbstractReportSchedulable.JOBDATAMAP_OUTPUT_FORMAT_KEY));
        String target = manifestData.get("target").toString().toUpperCase();
        infos.put(_COLUMN_TARGET, target);
        if (target.equals(PilotageReportTarget.ORGUNIT.name()))
        {
            String contextId = (String) manifestData.get(OrgUnitReportSchedulable.JOBDATAMAP_ORGUNIT_KEY);
            if (StringUtils.isNotEmpty(contextId))
            {
                infos.put(_COLUMN_CONTEXT, _getContentInfos((String) manifestData.get(OrgUnitReportSchedulable.JOBDATAMAP_ORGUNIT_KEY)));
            }
            infos.put(_COLUMN_CATALOG, _getCatalogInfos((String) manifestData.get(OrgUnitReportSchedulable.JOBDATAMAP_CATALOG_KEY)));
            infos.put(_COLUMN_LANG, _getLanguageInfos((String) manifestData.get(OrgUnitReportSchedulable.JOBDATAMAP_LANG_KEY)));
        }
        else if (target.equals(PilotageReportTarget.PROGRAM.name()))
        {
            String programId = manifestData.get(ProgramReportSchedulable.JOBDATAMAP_PROGRAM_KEY).toString();
            infos.put(_COLUMN_CONTEXT, _getContentInfos(programId));
            try
            {
                Program program = _resolver.resolveById(programId);
                infos.put(_COLUMN_CATALOG, _getCatalogInfos(program.getCatalog()));
                infos.put(_COLUMN_LANG, _getLanguageInfos(program.getLanguage()));
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn("The content '" + programId + "' has probably been deleted.");
            }
        }
        return infos;
    }
    
    private Map<String, Object> _getReportTypeInfos(String typeId)
    {
        Map<String, Object> infos = new HashMap<>();
        infos.put("value", typeId);
        
        PilotageReport report = _reportEP.getExtension(typeId);
        if (report != null)
        {
            infos.put("label", report.getLabel());
        }
        
        return infos;
    }
    
    private Map<String, Object> _getContentInfos(String contentId)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", contentId);
        
        try
        {
            Content content = _resolver.resolveById(contentId);
            infos.put("title", content.getTitle());
            infos.put("isSimple", _contentHelper.isSimple(content));
        }
        catch (UnknownAmetysObjectException e)
        {
            // Nothing
        }
        
        return infos;
    }
    
    private Map<String, Object> _getCatalogInfos(String name)
    {
        Map<String, Object> infos = new HashMap<>();
        infos.put("value", name);
        
        Catalog catalog = _catalogManager.getCatalog(name);
        if (catalog != null)
        {
            infos.put("label", catalog.getTitle());
        }
        return infos;
        
    }
    
    private Map<String, Object> _getLanguageInfos(String lang)
    {
        Map<String, Object> infos = new HashMap<>();
        infos.put("code", lang);
        
        if (lang != null)
        {
            Language language = _languageManager.getLanguage(lang);
            if (language != null)
            {
                infos.put("icon", language.getSmallIcon());
                infos.put("label", language.getLabel());
            }
        }
        
        return infos;
    }
    /**
     * Get the log file associated to the report
     * @param manifestData The manifest content
     * @return the log file if exists, or null
     */
    private File _getLogFile(Map<String, Object> manifestData)
    {
        PilotageReport report = _reportEP.getExtension((String) manifestData.get("type"));
        if (report != null)
        {
            File logFile = new File(_pilotageLogFileManager.getLogsDirectory(), report.getType() + "-" + manifestData.get("date") + ".log");
            if (logFile.exists())
            {
                return logFile;
            }
        }
        
        return null;
    }

    private List<Object> _getSortList(Object sortValues)
    {
        if (sortValues != null)
        {
            return _jsonUtils.convertJsonToList(sortValues.toString());
        }
        
        return null;
    }
    
    @SuppressWarnings("unchecked")
    private File[] _sortFiles(File[] files, List<Object> sortList)
    {
        if (sortList != null)
        {
            for (Object sortValueObj : Lists.reverse(sortList))
            {
                Map<String, Object> sortValue = (Map<String, Object>) sortValueObj;
                Comparator<File> comparator = _NAME_TO_COMPARATOR.get(sortValue.get("property"));
                Object direction = sortValue.get("direction");
                if (direction != null && direction.toString().equalsIgnoreCase("DESC"))
                {
                    comparator = Collections.reverseOrder(comparator);
                }
                Arrays.sort(files, comparator);
            }
        }
        
        return files;
    }
}
