/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * This controller is used for the steering workflow buttons of ODF contents
 */
Ext.define('Ametys.plugins.odf.pilotage.controller.PilotageStatusButtonController', {
    extend: 'Ametys.plugins.cms.content.controller.SmartContentController',
    
    /**
	 * @cfg {String} pilotage-status The pilotage status of the button (MENTION_VALIDATED, ORGUNIT_VALIDATED or CFVU_VALIDATED)
	 */
    /**
     * @property {String} status See #cfg-pilotage-status
     */
    
    /**
	 * @cfg {String} on-icon-decorator The CSS class for icon decorator when the pilotage status is on.
	 */
	/**
	 * @property {String} _onIconDecorator See #cfg-on-icon-decorator
	 * @private
	 */
    
    constructor: function(config)
    {
        this.callParent(arguments);
        
        this.status = this.getInitialConfig("pilotage-status");
        this._onIconDecorator = this.getInitialConfig("on-icon-decorator");
		
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
    },
    
    /**
     * @private
     * Listener handler for modified messages
     * @param {Ametys.message.Message} message the message
     */
    _onModified: function(message)
    {
        if (this.updateTargetsInCurrentSelectionTargets(message))
        {
            this.refresh();
        }
    },
    
    /**
	 * Refresh the controller from the content informations state of given targets
	 * @param targets The content targets
	 * @protected
	 */
	_getStatus: function (targets)
	{
		this.disable();
		
		var contentIds = [];
		for (var i=0; i < targets.length; i++)
		{
			contentIds.push(targets[i].getParameters().id);
		}
		
		this.serverCall ('getStatus', [contentIds, this.status], this._getStatusCb, { cancelCode: this.self.getName() + "$" + this.getId(), refreshing: true });
	},
    
    /**
	 * @protected
	 * Callback for the button reloading process
     * @param {Object} params the server's response
     */
	_getStatusCb: function (params)
	{
		this.callParent(arguments);
		
		this.toggle(params["isPressed"]);
		this._updatePilotageIcon(params);

		if (!this.isDisabled())
		{
			if (!params["isEnabled"])
			{
				this.disable();
			}
			
			this._updatePilotageTooltipDescription(params);
		}
		
	},
	
	/**
	 * @private
	 * Update the icons
	 * @param {Object} params The JSON result 
	 */
	_updatePilotageIcon: function (params)
	{
		if (params["isPressed"])
		{
			this.setIconDecorator(this._onIconDecorator);
		}
		else
		{
			this.setIconDecorator(null);
		}
	},
	
	/**
     * @private
     * Update the tooltip description according state of the current selection
     * @param {Object} params The JSON result received
     */
    _updatePilotageTooltipDescription: function(params)
    {
    	var description = "";
    	if (params["isPressed"])
    	{
    		if (params["isEnabled"])
			{
    			description += this.getInitialConfig("go-back-start-description");
    			description += params['go-back']; 
    			description += this.getInitialConfig("go-back-end-description");
			}
    		else
    		{
    			description += this.getInitialConfig("go-back-disabled-start-description");
    			description += params['go-back-disabled'];
    			description += this.getInitialConfig("go-back-disabled-end-description");
			}
    	}
    	else 
    	{
    		if (params["isEnabled"])
    	    {
    			description += this.getInitialConfig("go-next-start-description");
    			description += params['go-next'];
    			description += this.getInitialConfig("go-next-end-description");
			}
    		else
    		{
    			description += this.getInitialConfig("go-next-disabled-start-description");
    			description += params['go-next-disabled'];
    			description += this.getInitialConfig("go-next-disabled-end-description");
			}
    	}
    	
    	this.setDescription(description);
    }
});
