/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Helper to open a dialog box with date selection and comment for a step validation
 * @private
 */
Ext.define('Ametys.plugins.odf.pilotage.helper.ValidateDialog', {
	singleton: true,
	
    /**
     * Open dialog box with date and comment fields
     * @param {Object} config the dialog box configuration:
     * @param {String} config.dialogTitle The dialog title (required)
     * @param {String} config.dialogIconCls The css class for dialog icon (required)
     * @param {Function} config.validateCallback The callback function (required). The callback will be called with following arguments:
     * @param {String} config.validateCallback.date The date value
     * @param {String} config.validateCallback.comment The comment value
     * @param {String} [config.minDate] The minimum allowed date
     * @param {String} [config.minDateText] The error text to display when the date in is before the minDate
     * @param {String} [config.maxDate] The maximun allowed date
     * @param {String} [config.maxDateText] The error text to display when the date in is before the maxDate
     * @param {String} [config.dateValue] The date value to initialize the date field
     * @param {Boolean} [autoCloseDialog=true] Set to false to not close dialog after clicking on validate button
     * @return the dialog box
     */
	open: function (config)
    {
        config = Ext.applyIf(config, {
            autoCloseDialog: true
        });
        
        this._createDialog(config);
        
        this._dialog.show();
        
        return this._dialog;
    },
    
    _createFormPanel: function (config)
    {
        var dateConfig = {value: config.dateValue};
        
        if (config.minDate)
        {
            dateConfig.minValue = config.minDate;
            if (dateConfig.minDateText)
            {
                dateConfig: config.minDateText + " (" +  Ext.Date.format(config.minDate, "d/m/Y") + ")"
            }
        }
        
        if (config.maxDate)
        {
            dateConfig.maxValue = config.maxDate;
            if (dateConfig.maxDateText)
            {
                dateConfig: config.maxDateText + " (" +  Ext.Date.format(config.maxDate, "d/m/Y") + ")"
            }
        }
        
        return Ext.create('Ext.form.Panel', {
            defaults: {
                cls: 'ametys',
                labelSeparator: '',
                labelAlign: 'right',
                labelWidth: 120,
                width: '100%',
                msgTarget: 'side'
            },
            
            border: false,
            scrollable: false,
            
            items: [
                Ext.apply({
                    xtype: 'datefield',
                    name: 'date',
                    itemId: 'date',
                    allowBlank: false,
                    submitFormat: Ext.Date.patterns.ISO8601DateTime,
                    fieldLabel: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_VALIDATION_DIALOG_DATE_LABEL}} *",
                    ametysDescription: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_VALIDATION_DIALOG_DATE_DESC}}"
                }, dateConfig),
                {
                    xtype: 'textareafield',
                    name: 'comment',
                    itemId: 'comment',
                    fieldLabel: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_VALIDATION_DIALOG_COMMENT_LABEL}}",
                    ametysDescription: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_VALIDATION_DIALOG_COMMENT_DESC}}"
                }
            ]
        });
    },
    
    _createDialog: function(config)
    {
        this._form = this._createFormPanel(config);
        
        var items = [];
        if (config.hintMsg)
        {
            items.push({
                xtype: 'component',
                cls: 'a-text',
                html: config.hintMsg
            })
        }
        items.push(this._form);
        
        this._dialog = Ext.create('Ametys.window.DialogBox', {
            title: config.dialogTitle,
            iconCls: config.dialogIconCls,
            
            width: 450, 
            scrollable: false,
            
            items: items,
            
            referenceHolder: true,
            defaultButton: 'validate',
            closeAction: 'destroy',
            
            buttons: [{
                reference: 'validate',
                text: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_VALIDATION_DIALOG_OK_BTN}}",
                handler: Ext.bind(this._validateFn, this, [config]),
                scope: this
            }, {
                text: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_VALIDATION_DIALOG_CANCEL_BTN}}",
                handler : function () { this._dialog.close()},
                scope: this
            }]    
        });
        
        return this._dialog;
    },
    
    _validateFn: function (config)
    {
        var form = this._form.getForm();
        if (!form.isValid())
        {
            return;
        }
        
        var values = form.getValues();
        
        config.validateCallback(values.date, values.comment);
        
        if (config.autoCloseDialog)
        {
            this._dialog.close();
        }
    }
});
