/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Tool to edit MCC in course 
 * @private
 */
Ext.define('Ametys.plugins.odf.pilotage.tool.MCCCourseTool', {
    extend: 'Ametys.plugins.odf.pilotage.tool.ContentViewTreeGridTool',
   
    statics: {
        /**
         * @private
         * Fill a cache object with the table ref data
         */
        _getTableRefData: function(tableRefId, cache) {
            Ametys.data.ServerComm.send({
                plugin: 'cms',
                url: 'search/list.json',
                parameters: {
                    model: tableRefId
                },
                priority: Ametys.data.ServerComm.PRIORITY_MAJOR,
                errorMessage: true,
                callback: {
                    handler: function(response) {
                        if (Ametys.data.ServerComm.isBadResponse(response))
                        {
                            return;
                        }
                        
                        let result = Ext.JSON.decode(Ext.dom.Query.selectValue("", response));
                        for (let c of result.contents)
                        {
                            cache[c.id] = {
                                title: Ametys.plugins.cms.search.SearchGridHelper.getDefaultLocalizedValue(c.title),
                                shortLabel: c.properties.shortLabel
                            };
                        }
                    },
                    scope: this,
                },
                responseType: 'text'
            });
        },
        
        /**
         * @private
         * @property {Object} _natureCache The internal reftable cache 
         */
        _natureCache: {},
        
        /**
         * @private
         * @property {Object} _regimeCache The internal reftable cache 
         */
        _regimeCache: {},
        
        /**
         * @private
         * @property {Object} _modalite2Cache The internal reftable cache 
         */
        _modaliteCache: {},
        
        /**
         * @private
         * @property {Ext.XTemplate} _repeaterTpl The template used for rendering repeater entries
         */
        _repeaterTpl: Ext.create('Ext.XTemplate', [
            '<span data-qtip="{tip}"><tpl for="entries">',
                '<tpl if="position &gt; 1">',
                    ", ",
                '</tpl>',
                '<span class="a-grid-repeater-entry">{label}</span>',
            '</tpl></span>'
        ]),
        
        renderIsEvaluated: function(value, metaData, record, rowIndex, colIndex, store, view, dataIndex)
        {
            let v = value;
            if ((v === undefined || v == null)
                && (record.get('contenttypesIds') || []).includes('org.ametys.plugins.odf.Content.course'))
            {
                v = true;
            }
            return Ametys.plugins.cms.search.SearchGridHelper.renderBooleanIcon(v, metaData, record, rowIndex, colIndex, store, view, dataIndex)
        },
        
        renderMCCRegime: function(value, metaData, record, rowIndex, colIndex, store, view, dataIndex)
        {
            return this._readShortLabelOrTitle(record.data, "mccRegime", Ametys.plugins.odf.pilotage.tool.MCCCourseTool._regimeCache, "");
        },
        
        /**
         * @private
         * Get the short label if available in cache, title otherwise
         */
        _readShortLabelOrTitle(values, dataindex, cache, otherwise)
        {
            if (values[dataindex])
            {
                let cacheEntry = cache[values[dataindex]];
                if (cacheEntry)
                {
                    return cacheEntry.shortLabel || cacheEntry.title;
                }
                else
                {
                    return values[dataindex + "_content"][values[dataindex]].title;
                }
            }
            else
            {
                return otherwise;
            }
        },
        
        renderRepeater: function(value, metaData, record, rowIndex, colIndex, store, view, dataIndex)
        {
            if (Ext.isEmpty(value))
            {
                return '';
            }
        
            let repeater = record.get(dataIndex + "_repeater") || record.get(dataIndex);
            if (!Ext.isObject(repeater))
            {
                return '';
            }
            
            let rawValues = repeater.entries;
            if (!Ext.isArray(rawValues) || rawValues.length == 0)
            {
                return '';
            }
            
            let labels = [];
            
            let entries = [];
            for (let i in rawValues)
            {
                let rawValue = rawValues[i];
                
                let label = "";
                
                label += this._readShortLabelOrTitle(rawValue.values, "nature", Ametys.plugins.odf.pilotage.tool.MCCCourseTool._natureCache, "");
                
                label += " (";

                label += this._readShortLabelOrTitle(rawValue.values, "modalite", Ametys.plugins.odf.pilotage.tool.MCCCourseTool._modaliteCache, "");
                
                if (rawValue.values["coefficient"])
                {
                    label += ", Coef. " + rawValue.values["coefficient"];
                }

                label += ")";

                
                labels.push(label);
                entries.push({
                    label: label,
                    position: rawValue.position
                });
                
            }
            
            let html = this._repeaterTpl.applyTemplate({
                entries: entries,
                tip: labels.join(",<br/>")
            });
            
            return html;
        }
    },
    
    _createTree: function()
    {
        return Ext.create('Ametys.plugins.odf.pilotage.tool.MCCCourseTreeGridPanel', this._getTreeConfig());
    },
   
    _getColumnsCb: function(columnsData, args)
    {
        let columnsDataModified = columnsData.map(c => {
            if (c.type == "repeater" && (c.path == "mccSession1" || c.path == "mccSession2"))
            {
                c.widget = "edition.mcccourserepeater";
                c['widget-params'] = {
                    dataIndexes: ["mccSession1", "mccSession2"],
                    authorizeCopy: "reportSession2",
                    doNotCopy: ["modalite", "modalite_content", "modalite_content_initial"]
                };
            } 
            return c;
        });
        
        this.callParent([columnsDataModified, args]);
    },
    
    _getTreeConfig: function()
    {
        var treeCfg = this.callParent(arguments);
        return Ext.apply(treeCfg, {
            forceReadonlyField: "mccSession2"
        });
    }
}); 

// Fetch all renderer values
(function () {
    Ametys.plugins.odf.pilotage.tool.MCCCourseTool._getTableRefData('reference-table-search-ui.odf-enumeration.MccSessionNature', Ametys.plugins.odf.pilotage.tool.MCCCourseTool._natureCache);
    Ametys.plugins.odf.pilotage.tool.MCCCourseTool._getTableRefData('reference-table-search-ui.odf-enumeration.MccRegime', Ametys.plugins.odf.pilotage.tool.MCCCourseTool._regimeCache);
    Ametys.plugins.odf.pilotage.tool.MCCCourseTool._getTableRefData('reference-table-search-ui.odf-enumeration.MccModaliteSession1', Ametys.plugins.odf.pilotage.tool.MCCCourseTool._modaliteCache);
    Ametys.plugins.odf.pilotage.tool.MCCCourseTool._getTableRefData('reference-table-search-ui.odf-enumeration.MccModaliteSession2', Ametys.plugins.odf.pilotage.tool.MCCCourseTool._modaliteCache);
})();
