/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Select an output format for report.
 */
Ext.define('Ametys.plugins.odfpilotage.widget.SelectReportOutputFormat', {
    extend: 'Ametys.form.widget.ComboBox',
    
    /**
     * @private
     * @property {String} _extensionId The report extension ID
     */
    _extensionId: null,
    
    constructor: function (config)
    {
        config = Ext.apply(config, {
            queryMode: 'local',
            valueField : 'id',
            displayField : 'label'
        });
        
        this.callParent(arguments);
        
        if (this.form && Ext.isFunction(this.form.onRelativeFieldsChange))
        {
            var prefix = config.name.split('$')[0];
            this.form.onRelativeFieldsChange([prefix + '$extensionId'], this, this._onExtensionIdFieldChange);
        }
        
        this._extensionId = config.extensionId;
    },
    
    getStoreCfg: function(config)
    {
        return {
            proxy: {
                type: 'ametys',
                role: 'org.ametys.plugins.odfpilotage.report.ReportOutputFormatHelper',
                methodName: 'getSupportedOutputFormats',
                methodArguments: ['extensionId'],
                reader: {
                    type: 'json',
                    rootProperty: "data"
                }
            },
            autoLoad: true, 
            sorters: [{property: config.displayField, direction: 'ASC'}],
            listeners: {
                scope: this,
                beforeload: function(store, operation)
                {
                    // Get the parameters, or create empty object if needed
                    var parameters = operation.getParams() || {};
                    // Add the enumerator role to the parameters
                    params = Ext.apply(parameters, {extensionId: this._extensionId});
                    // Set the parameters
                    operation.setParams(parameters);
                },
                load: function(store, records)
                {
                    var defaultValues = records.filter(r => r.data.isDefault);
                    if (defaultValues.length > 0)
                    {
                        this.setValue(defaultValues[0]);
                    }
                }
            }
        };
    },
    
    /**
     * @private
     * Listener when the field holding the extension id change
     * @param {Ext.form.Field} field The extension id field
     */
    _onExtensionIdFieldChange: function(field)
    {
        var value = field.getValue() || field.value;
        
        // in case of externalizable field
        value = Ametys.form.widget.Externalizable.getValueInUse(value);
        
        if (value != this._extensionId)
        {
            this._extensionId = value;
            this.getStore().load();
        }
    }
});