/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.helper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;

import com.opensymphony.workflow.WorkflowException;

/**
 * This component handle the content of the micro skill tool
 */
public class MicroSkillsTreeHelper extends ODFContentsTreeHelper implements Contextualizable
{
    /** The right for course edit */
    public static final String COURSE_EDIT_RIGHT_ID = "ODF_Rights_Course_Edit";
    /** The right for program edit */
    public static final String PROGRAM_EDIT_RIGHT_ID = "ODF_Rights_Program_Edit";
    /** The INEXISTING SKILL warning */
    public static final String INEXISTING_SKILL_WARN = "INEXISTING_SKILL";
    /** The NOT IN PROGRAM warning */
    public static final String NOT_IN_PROGRAM_WARN = "NOT_IN_PROGRAM";
    /** The edit action ID */
    public static final int EDIT_ACTION_ID = 2;
    /** The context */
    protected Context _context;
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The right manager */
    protected RightManager _rightManager;
    /** The content workflow helper*/
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    /**
     * Get the columns
     * @param programId The id of the selected program
     * @return The skills columns
     */
    @Callable (rights = "ODF_Rights_Pilotage_MICROSKILLS")
    public Map<String, Object> getSkillsColumns(String programId)
    {
        Map<String, Object> skills = new LinkedHashMap<>();
        
        if (StringUtils.isNotBlank(programId))
        {
            Program program = _ametysResolver.resolveById(programId);
            
            List<ContentValue> macroSkills = new ArrayList<>();
            
            ContentValue[] contentMacroSkills = program.getValue("skills", true);
            if (contentMacroSkills != null && contentMacroSkills.length > 0)
            {
                macroSkills.addAll(Arrays.asList(contentMacroSkills));
            }
            
            ContentValue[] transversalSkills = program.getValue("transversalSkills", true);
            if (transversalSkills != null && transversalSkills.length > 0)
            {
                macroSkills.addAll(Arrays.asList(transversalSkills));
            }
            
            Locale locale = Locale.of(StringUtils.defaultIfBlank((String) ContextHelper.getRequest(_context).getAttribute("locale"), "fr"));
            // Get the macro skills as Json columns
            for (ContentValue macroSkillContentValue : macroSkills)
            {
                Optional<ModifiableContent> optionalContent = macroSkillContentValue.getContentIfExists();
                if (optionalContent.isPresent())
                {
                    Map<String, Object> macroSkillInfos = _macroSkillToColumnJson(optionalContent.get(), locale);
                    skills.put(macroSkillContentValue.getContentId(), macroSkillInfos);
                }
            }
        }
        
        return skills;
    }
    
    private Map<String, Object> _skillToColumnJson(Content skill, Locale locale)
    {
        Map<String, Object> skillInfos = new HashMap<>();
        
        skillInfos.put("id", skill.getId());
        skillInfos.put("code", skill.getValue("code"));
        skillInfos.put("skillCode", skill.getValue("skillCode"));
        skillInfos.put("title", skill.getTitle(locale));

        return skillInfos;
    }
    
    private Map<String, Object> _macroSkillToColumnJson(Content macroSkill, Locale locale)
    {
        // Retrieve the info of the macro skill
        Map<String, Object> skillInfos = _skillToColumnJson(macroSkill, locale);
        
        // If the macro skill has micro skills, also retrieve them
        ContentValue[] microSkills = macroSkill.getValue("microSkills");
        
        if (microSkills != null)
        {
            Map<String, Object> microSkillsInfos = new LinkedHashMap<>();
            for (ContentValue microSkill : microSkills)
            {
                Optional<ModifiableContent> optionalContent = microSkill.getContentIfExists();
                if (optionalContent.isPresent())
                {
                    Content microSkillContent = optionalContent.get();
                    String microSkillId = microSkillContent.getId();
                    Map<String, Object> microSkillInfo = _skillToColumnJson(microSkillContent, locale);
                    
                    microSkillsInfos.put(microSkillId, microSkillInfo);
                }
            }
            
            skillInfos.put("microSkills", microSkillsInfos);
        }
        
        return skillInfos;
    }
    
    @Override
    protected Map<String, Object> content2Json(Content content, List<String> path)
    {
        Map<String, Object> infos = super.content2Json(content, path);

        String programId = path.get(0);
        infos.computeIfAbsent("notEditableDataIndex", l -> new ArrayList<>());

        // If the content is a course, we need to check if it is editable and if it has micro skills for the program
        if (content instanceof Course course)
        {
            if (!_contentWorkflowHelper.isAvailableAction(course, EDIT_ACTION_ID))
            {
                infos.put("notEditableData", true);
            }

            // Retrieve the acquired micro skills for the program
            ContentValue[] acquiredMicroSkillsValues = course.getAcquiredSkills(programId);
            if (acquiredMicroSkillsValues != null)
            {
                Map<String, Object> acquiredMicroSkills = Arrays.stream(acquiredMicroSkillsValues)
                        .map(value -> value.getContentId())
                        .collect(Collectors.toMap(value -> value,
                                value -> true)
                                );
                
                // If the course has microskills for the program, set them in the result
                if (acquiredMicroSkills != null)
                {
                    infos.putAll(acquiredMicroSkills);
                }
            }
        }
        // If the content is a program, we need to check if it is editable and if it has blocking micro skills
        else if (content instanceof Program program && program.hasValue("blockingMicroSkills"))
        {
            if (!_contentWorkflowHelper.isAvailableAction(program, EDIT_ACTION_ID))
            {
                infos.put("notEditableData", true);
            }
            
            if (program.hasValue("blockingMicroSkills"))
            {
                Map<String, Boolean> blockingMicroSkills = Arrays.stream((ContentValue[]) program.getValue("blockingMicroSkills"))
                    .map(value -> value.getContentId())
                    .collect(Collectors.toMap(value -> value, value -> true));
                
                infos.putAll(blockingMicroSkills);
            }
        }
        // If the content is not a Course nor the root program, it is not editable
        else if (!path.get(0).equals(content.getId()))
        {
            infos.put("notEditableData", true);
        }
        
        return infos;
    }
    
    /**
     * Save edition on the grid
     * @param contentId The id of the content to edit
     * @param changes The changes required
     * @param programRootId The id of the program at root
     * @return The result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> saveEdition(String contentId, Map<String, Object> changes, String programRootId)
    {
        Content content =  _ametysResolver.resolveById(contentId);
        if (content instanceof Course course)
        {
            return _saveEditionForCourse(programRootId, course, changes);
        }
        else if (content instanceof Program program)
        {
            return _saveEditionForProgram(program, changes);
        }
        else
        {
            return Map.of();
        }
    }
    
    private Map<String, Object> _saveEditionForCourse(String programRootId, Course course, Map<String, Object> changes)
    {
        Map<String, Object> result = new HashMap<>();
        
        Program program = _resolver.resolveById(programRootId);
        Set<String> programMicroSkillsIds = _getMicroSkills(program);
        
        if (!_rightManager.currentUserHasRight(COURSE_EDIT_RIGHT_ID, course).equals(RightResult.RIGHT_ALLOW))
        {
            result.put("success", false);
            result.put("errorMsg", "unauthorized");
            return result;
        }
        
        ModifiableModelAwareRepeater acquiredMicroSkillsRepeater = course.getValue("acquiredMicroSkills");
        
        Map<String, Map<String, Object>> skillsByProgramId = _getCourseSkillsByProgramAsMap(acquiredMicroSkillsRepeater);
        
        Map<String, Object> entryForProgram = skillsByProgramId.computeIfAbsent(programRootId, entry -> new HashMap<>(Map.of("program", programRootId)));
        
        @SuppressWarnings("unchecked")
        List<String> currentMicroSkillsIds = (List<String>) entryForProgram.computeIfAbsent("microSkills", microSkills -> new ArrayList<>());
        
        for (String microSkillId : changes.keySet())
        {
            // The skill is valid in the Program, so we can do the edition
            if (_checkValidSkills(microSkillId, programMicroSkillsIds, result))
            {
                boolean isMicroSkillSelected = (boolean) changes.get(microSkillId);
                
                // The microskill is not selected so we need to remove it
                if (currentMicroSkillsIds.contains(microSkillId) && !isMicroSkillSelected)
                {
                    currentMicroSkillsIds.remove(microSkillId);
                }
                // If the microskill was not contained in the current microskills and it is now selected, add it
                else if (!currentMicroSkillsIds.contains(microSkillId) && isMicroSkillSelected)
                {
                    currentMicroSkillsIds.add(microSkillId);
                }
            }
        }
        
        // If there are no microskills for the program anymore, remove the repeater entry altogether
        if (currentMicroSkillsIds.isEmpty())
        {
            skillsByProgramId.remove(programRootId);
        }
        
        try
        {
            // Edit the course with the new micro skill for the program
            Map<String, Object> inputsForEditCourse = new HashMap<>();
              
            inputsForEditCourse.put("acquiredMicroSkills", new ArrayList<>(skillsByProgramId.values()));
            
            _contentWorkflowHelper.editContent(course, inputsForEditCourse, EDIT_ACTION_ID);
        }
        catch (AmetysRepositoryException | WorkflowException e)
        {
            result.put("success", false);
            result.put("errorMsg",  e.getMessage());
        }

        // If there are no errors, then set success to true
        if (!result.containsKey("errorMsg"))
        {
            result.put("success", true);
        }
        
        return result;
    }
    
    private Content _getIfStillExists(String microSkill)
    {
        try
        {
            return _resolver.resolveById(microSkill);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }

    private Map<String, Map<String, Object>> _getCourseSkillsByProgramAsMap(ModifiableModelAwareRepeater courseAcquiredMicroSkills)
    {
        Map<String, Map<String, Object>> result = new HashMap<>();
        if (courseAcquiredMicroSkills != null)
        {
            List<? extends ModelAwareRepeaterEntry> skillByProgramEntries = courseAcquiredMicroSkills.getEntries();
            for (ModelAwareRepeaterEntry skillByProgramEntry : skillByProgramEntries)
            {
                Map<String, Object> entry = new HashMap<>();
                ContentValue contentValueProgram = skillByProgramEntry.getValue(Course.ACQUIRED_MICRO_SKILLS_PROGRAM);
                
                List<String> microSkillsIds = new ArrayList<>();
                ContentValue[] microSkillsContentValues = skillByProgramEntry.getValue(Course.ACQUIRED_MICRO_SKILLS_SKILLS);
                for (ContentValue microSkillContentValue : microSkillsContentValues)
                {
                    microSkillsIds.add(microSkillContentValue.getContentId());
                }
                
                entry.put("program", contentValueProgram.getContentId());
                entry.put("microSkills", microSkillsIds);
                
                result.put(contentValueProgram.getContentId(), entry);
            }
        }
        
        return result;
    }
    
    private Map<String, Object> _saveEditionForProgram(Program program, Map<String, Object> changes)
    {
        Map<String, Object> result = new HashMap<>();
        
        if (!_rightManager.currentUserHasRight(PROGRAM_EDIT_RIGHT_ID, program).equals(RightResult.RIGHT_ALLOW))
        {
            result.put("success", false);
            result.put("errorMsg", "unauthorized");
            return result;
        }
        
        Set<String> programMicroSkillsIds = _getMicroSkills(program);
        
        List<String> blockingMicroSkillsToSet = new ArrayList<>();
        if (program.hasValue("blockingMicroSkills"))
        {
            blockingMicroSkillsToSet.addAll(Arrays.stream((ContentValue[]) program.getValue("blockingMicroSkills"))
                    .map(value -> value.getContentId())
                    .toList());
        }
        
        for (String microSkillId : changes.keySet())
        {
            // The skill is valid in the Program, so we can do the edition
            if (_checkValidSkills(microSkillId, programMicroSkillsIds, result))
            {
                boolean isMicroSkillSelected = (boolean) changes.get(microSkillId);
                // The microskill is not selected so we need to remove it
                if (blockingMicroSkillsToSet.contains(microSkillId) && !isMicroSkillSelected)
                {
                    blockingMicroSkillsToSet.remove(microSkillId);
                }
                // If the microskill was not contained in the current microskills and it is now selected, add it
                else if (!blockingMicroSkillsToSet.contains(microSkillId) && isMicroSkillSelected)
                {
                    blockingMicroSkillsToSet.add(microSkillId);
                }
            }
        }
        
        try
        {
            // Edit the program with the new blocking micro skill
            Map<String, Object> inputsForEditProgram = new HashMap<>();
              
            inputsForEditProgram.put("blockingMicroSkills", blockingMicroSkillsToSet);
            
            _contentWorkflowHelper.editContent(program, inputsForEditProgram, EDIT_ACTION_ID);
        }
        catch (AmetysRepositoryException | WorkflowException e)
        {
            result.put("success", false);
            result.put("errorMsg", e.getMessage());
        }
        
        // If there are no errors, then set success to true
        if (!result.containsKey("errorMsg"))
        {
            result.put("success", true);
        }
        
        return result;
    }
    
    private Set<String> _getMicroSkills(Program program)
    {
        Set<String> programMicroSkillsIds = new HashSet<>();
        programMicroSkillsIds.addAll(program.getTransversalSkills()
                .stream()
                .map(macroSkill -> macroSkill.getValue("microSkills", false, new ContentValue[0]))
                .flatMap(Arrays::stream)
                .map(ContentValue::getContentId)
                .toList());
        programMicroSkillsIds.addAll(program.getSkills()
                .stream()
                .map(macroSkill -> macroSkill.getValue("microSkills", false, new ContentValue[0]))
                .flatMap(Arrays::stream)
                .map(ContentValue::getContentId)
                .toList());
        
        return programMicroSkillsIds;
    }
    
    private boolean _checkValidSkills(String microSkillId, Set<String> programMicroSkillsIds,  Map<String, Object> result)
    {
        Content microSkill = _getIfStillExists(microSkillId);
        // The micro skill is invalid for the program if it does not exist anymore (a deleted skill) or if it is not contained in the skills of the program
        if (microSkill == null)
        {
            @SuppressWarnings("unchecked")
            Map<String, Object> warnings = (Map<String, Object>) result.computeIfAbsent("warnings", e -> new HashMap<>());
            warnings.put(INEXISTING_SKILL_WARN, true);
            
            @SuppressWarnings("unchecked")
            Set<String> uncommitedChanges = (Set<String>) warnings.computeIfAbsent("notModifiedSkills", e -> new HashSet<>());
            uncommitedChanges.add(microSkillId);
            
            return false;
        }
        else if (!programMicroSkillsIds.contains(microSkillId))
        {
            @SuppressWarnings("unchecked")
            Map<String, Object> warnings = (Map<String, Object>) result.computeIfAbsent("warnings", e -> new HashMap<>());
            @SuppressWarnings("unchecked")
            Set<String> notInProgram = (Set<String>) warnings.computeIfAbsent(NOT_IN_PROGRAM_WARN, k -> new HashSet<>());
            notInProgram.add("'" + microSkill.getTitle() + "' [" + microSkill.getValue("code") + "]");
            
            @SuppressWarnings("unchecked")
            Set<String> uncommitedChanges = (Set<String>) warnings.computeIfAbsent("notModifiedSkills", e -> new HashSet<>());
            uncommitedChanges.add(microSkillId);
            
            return false;
        }
        
        return true;
    }
}
