/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
     */
package org.ametys.plugins.odfpilotage.report;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Helper for report output format.
 */
public class ReportOutputFormatHelper implements Component, Serviceable
{
    /** The report extension point */
    protected ReportExtensionPoint _reportEP;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _reportEP = (ReportExtensionPoint) smanager.lookup(ReportExtensionPoint.ROLE);
    }
    
    /**
     * Get the output format for a given report extension id.
     * @param extensionId The report extension ID. Can be null.
     * @return the list of output formats with id, label and isDefault
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public List<Map<String, Object>> getSupportedOutputFormats(String extensionId)
    {
        if (extensionId == null || !_reportEP.hasExtension(extensionId))
        {
            return List.of();
        }
        
        PilotageReport report = _reportEP.getExtension(extensionId);
        String defaultOutputFormat = report.getDefaultOutputFormat();
        
        return _reportEP.getExtension(extensionId)
            .getSupportedOutputFormats()
            .stream()
            .map(
                format -> Map.of(
                    "id", format,
                    "label", new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_EXPORT_HELPER_OUTPUT_FORMAT_" + format.toUpperCase()),
                    "isDefault", format.equals(defaultOutputFormat)
                )
            )
            .toList();
    }
}
