/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections4.SetUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.content.compare.ContentComparator;
import org.ametys.cms.content.compare.ContentComparatorResult;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.plugins.odfpilotage.report.impl.tree.ProgramItemTree;
import org.ametys.plugins.odfpilotage.report.impl.tree.ProgramItemTree.ChangeType;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewHelper;
import org.ametys.runtime.model.ViewItemAccessor;

/**
 * The compare MCC catalog report (based on MCC report).
 */
public class MCCDiffReport extends AbstractMCCReport
{
    /** The key for the old catalog */
    public static final String PARAMETER_CATALOG_OLD = "catalogOld";

    private static final String __OLD_FOR_NEW_PROGRAM_ITEMS_CACHE_ID = MCCDiffReport.class.getName() + "$old4newProgramItems";

    private ContentComparator _contentComparator;
    private AbstractCacheManager _cacheManager;
    
    @Override
    protected String getType(Map<String, String> reportParameters)
    {
        return "mccdiff";
    }
    
    @Override
    protected boolean isGeneric()
    {
        return false;
    }

    @Override
    protected boolean isCompatibleSchedulable(AbstractReportSchedulable schedulable)
    {
        return schedulable.getId().equals("org.ametys.plugins.odfpilotage.schedulable.OrgUnitMCCDiffReportSchedulable");
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentComparator = (ContentComparator) manager.lookup(ContentComparator.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        super.initialize();
        
        _cacheManager.createRequestCache(__OLD_FOR_NEW_PROGRAM_ITEMS_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_OLD_FOR_NEW_PROGRAM_ITEMS_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_OLD_FOR_NEW_PROGRAM_ITEMS_DESCRIPTION"),
                true);
    }
    
    @Override
    protected ProgramItemTree createMCCTreeFromProgramItem(ProgramItem programItem, Map<String, String> reportParameters)
    {
        ProgramItem oldProgramItem = _findOldProgramItem(programItem, reportParameters.get(PARAMETER_CATALOG_OLD));
        ContentComparatorResult changes = _getObjectChanges(oldProgramItem, programItem);
        ProgramItemTree programItemTree = new ProgramItemTree(programItem, changes);
        populateMCCTree(programItemTree);
        return programItemTree;
    }
    
    @Override
    protected void populateMCCTree(ProgramItemTree tree)
    {
        ProgramItem oldProgramItem = (ProgramItem) tree.getChange().getContent1();
        ProgramItem newProgramItem = (ProgramItem) tree.getChange().getContent2();
        
        if (oldProgramItem == null)
        {
            // Only new
            List<ProgramItem> children = _odfHelper.getChildProgramItems(newProgramItem);
            for (ProgramItem child : children)
            {
                ContentComparatorResult changes = _getObjectChanges(null, child);
                ProgramItemTree childTree = tree.addChild(child, changes);
                populateMCCTree(childTree);
            }
        }
        else if (newProgramItem == null)
        {
            // Only old
            List<ProgramItem> children = _odfHelper.getChildProgramItems(oldProgramItem);
            for (ProgramItem child : children)
            {
                ContentComparatorResult changes = _getObjectChanges(child, null);
                ProgramItemTree childTree = tree.addChild(child, changes);
                populateMCCTree(childTree);
            }
        }
        else
        {
            List<ProgramItem> oldChildren = _odfHelper.getChildProgramItems(oldProgramItem);
            List<ProgramItem> newChildren = _odfHelper.getChildProgramItems(newProgramItem);

            // Add new program items
            for (ProgramItem newChild : newChildren)
            {
                ProgramItem currentOldChild = null;
                Class<? extends ProgramItem> newChildClass = newChild.getClass();
                String newChildCode = newChild.getCode();
                for (ProgramItem oldChild : oldChildren)
                {
                    if (oldChild.getCode().equals(newChildCode) && oldChild.getClass().equals(newChildClass))
                    {
                        currentOldChild = oldChild;
                        // Decrease the size of old children to do the next operations faster
                        oldChildren.remove(oldChild);
                        break;
                    }
                }

                ContentComparatorResult changes = _getObjectChanges(currentOldChild, newChild);
                ProgramItemTree childTree = tree.addChild(newChild, changes);
                populateMCCTree(childTree);
            }
            
            // Add old program items
            for (ProgramItem oldChild : oldChildren)
            {
                ContentComparatorResult changes = _getObjectChanges(oldChild, null);
                ProgramItemTree childTree = tree.addChild(oldChild, changes);
                populateMCCTree(childTree);
            }
        }
    }

    private ContentComparatorResult _getObjectChanges(ProgramItem oldProgramItem, ProgramItem newProgramItem)
    {
        return new ContentComparatorResult((Content) oldProgramItem, (Content) newProgramItem);
    }
    
    /**
     * Find the equivalent content in the new catalog
     * @param <T> Type of content to find
     * @param content Content in the current catalog
     * @return New equivalent content or null
     */
    private <T extends ProgramItem> T _findOldProgramItem(T content, String oldCatalog)
    {
        Cache<String, T> cache = _cacheManager.get(__OLD_FOR_NEW_PROGRAM_ITEMS_CACHE_ID);
        return cache.get(content.getId(), __ -> _odfHelper.getProgramItem(content, oldCatalog, content.getLanguage()));
    }
    
    @Override
    protected boolean isExcludedFromMCC(ProgramItemTree tree)
    {
        Course oldCourse = (Course) tree.getChange().getContent1();
        Course newCourse = (Course) tree.getChange().getContent2();
        
        // Is excluded if both are excluded or only one is excluded and the other does not exist
        return (oldCourse == null || _pilotageHelper.isExcludedFromMCC(oldCourse)) && (newCourse == null || _pilotageHelper.isExcludedFromMCC(newCourse));
    }
    
    @Override
    protected void saxSession(ContentHandler handler, ProgramItemTree tree, String sessionName) throws SAXException
    {
        ContentComparatorResult changes = tree.getChange();
        Content oldContent = changes.getContent1();
        Content newContent = changes.getContent2();
        
        ModelItem sessionDefinition = _sessionsView.getModelViewItem(sessionName).getDefinition();
        
        boolean oldContentSessionEnabled = oldContent != null && !_disableConditionsEvaluator.evaluateDisableConditions(sessionDefinition, sessionName, oldContent);
        boolean newContentSessionEnabled = newContent != null && !_disableConditionsEvaluator.evaluateDisableConditions(sessionDefinition, sessionName, newContent);
        
        // Only sax if session is enabled
        if (oldContentSessionEnabled || newContentSessionEnabled)
        {
            XMLUtils.startElement(handler, sessionName);
            
            // Re-compose the educational path of the old item from the new one
            String oldCatalog = Optional.ofNullable(oldContent)
                    .map(ProgramItem.class::cast)
                    .map(ProgramItem::getCatalog)
                    .orElse(null);
            List<ProgramItem> newPath = tree.getPath().getProgramItems(_resolver);
            List<ProgramItem> oldPath = _buildOldPath(newPath, oldCatalog);
            
            // Entries are grouped by teaching nature (nature can be empty)
            Map<String, List<ModelAwareRepeaterEntry>> oldEntriesByNature = oldContentSessionEnabled ? _getEntriesByNature(oldContent, sessionName, oldPath) : Map.of();
            Map<String, List<ModelAwareRepeaterEntry>> newEntriesByNature = newContentSessionEnabled ? _getEntriesByNature(newContent, sessionName, newPath) : Map.of();
            
            Set<String> natures = SetUtils.union(oldEntriesByNature.keySet(), newEntriesByNature.keySet());
            
            for (String nature : natures)
            {
                List<ModelAwareRepeaterEntry> oldEntries = oldEntriesByNature.getOrDefault(nature, List.of());
                List<ModelAwareRepeaterEntry> newEntries = newEntriesByNature.getOrDefault(nature, List.of());
                
                int oldEntriesSize = oldEntries.size();
                int newEntriesSize = newEntries.size();
                int maxSize = Integer.max(oldEntriesSize, newEntriesSize);
                
                for (int i = 0; i < maxSize; i++)
                {
                    if (i >= oldEntriesSize)
                    {
                        /* Added entry */
                        _saxSessionEntry(handler, newEntries.get(i), newContent, ChangeType.ADDED);
                    }
                    else if (i >= newEntriesSize)
                    {
                        /* Removed entry */
                        _saxSessionEntry(handler, oldEntries.get(i), oldContent, ChangeType.REMOVED);
                    }
                    else
                    {
                        ModelAwareRepeaterEntry oldEntry = oldEntries.get(i);
                        ModelAwareRepeaterEntry newEntry = newEntries.get(i);
                        
                        ContentComparatorResult result = _contentComparator.compare(oldEntry, newEntry, (ViewItemAccessor) ViewHelper.getModelViewItem(_sessionsView, sessionName));
                        if (result.areEquals())
                        {
                            /* No changes */
                            _saxSessionEntry(handler, newEntry, newContent, null);
                        }
                        else
                        {
                            /* Modified entries */
                            _saxSessionEntry(handler, oldEntry, oldContent, ChangeType.MODIFIED_OLD);
                            _saxSessionEntry(handler, newEntry, newContent, ChangeType.MODIFIED_NEW);
                        }
                    }
                }
            }
            
            XMLUtils.endElement(handler, sessionName);
        }
    }
    
    private void _saxSessionEntry(ContentHandler handler, ModelAwareRepeaterEntry sessionEntry, Content rootContent, ChangeType changeType) throws SAXException
    {
        // start entry
        AttributesImpl entryAttrs = new AttributesImpl();
        entryAttrs.addCDATAAttribute("name", String.valueOf(sessionEntry.getPosition()));
        if (changeType != null)
        {
            entryAttrs.addCDATAAttribute("changeType", changeType.name().toLowerCase());
        }
        XMLUtils.startElement(handler, "entry", entryAttrs);
        
        saxSessionEntryDetails(handler, sessionEntry, rootContent);
        
        // end entry
        XMLUtils.endElement(handler, "entry");
    }
    
    private List<ProgramItem> _buildOldPath(List<ProgramItem> newPath, String oldCatalog)
    {
        // Old item may not exists
        if (oldCatalog == null)
        {
            return List.of();
        }
        
        List<ProgramItem> oldPath = new ArrayList<>();
        for (ProgramItem newProgramItem : newPath)
        {
            ProgramItem oldProgramItem = _findOldProgramItem(newProgramItem, oldCatalog);
            // If old program item does not exists or oldProgramItem and newProgramItem
            // are the same (iterate on an old path because of structure modifications),
            // return an empty path
            if (oldProgramItem == null || oldProgramItem.equals(newProgramItem))
            {
                return List.of();
            }
            oldPath.add(oldProgramItem);
        }
        return oldPath;
    }
    
    private Map<String, List<ModelAwareRepeaterEntry>> _getEntriesByNature(Content content, String sessionName, List<ProgramItem> educationalPath)
    {
        return Optional.ofNullable(content)
                .map(c -> c.getRepeater(sessionName))
                .map(r -> {
                    List<EducationalPath> educationalPaths = educationalPath.isEmpty() ? List.of() : _odfHelper.getEducationPathFromPath(educationalPath);
                    return _odfHelper.getRepeaterEntriesByPath(r, educationalPaths);
                })
                .orElseGet(Stream::of)
                .collect(Collectors.<ModelAwareRepeaterEntry, String>groupingBy(this::_getNatureEnseignementId));
    }
    
    private String _getNatureEnseignementId(ModelAwareRepeaterEntry entry)
    {
        return Optional.of(entry)
                .map(e -> e.<ContentValue>getValue("natureEnseignement"))
                .map(ContentValue::getContentId)
                .orElse(StringUtils.EMPTY);
    }
    
    @Override
    protected void saxGlobalInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        super.saxGlobalInformations(handler, programItem, reportParameters);
        
        String oldCatalog = reportParameters.get(PARAMETER_CATALOG_OLD);
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", oldCatalog);
        XMLUtils.createElement(handler, "catalogOld", attrs, _catalogsManager.getCatalog(oldCatalog).getTitle());
    }
}
