/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.RichTextAttributeDefinition;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfpilotage.report.impl.tree.ProgramItemTree;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelViewItem;
import org.ametys.runtime.model.ModelViewItemGroup;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.disableconditions.DefaultDisableConditionsEvaluator;
import org.ametys.runtime.model.disableconditions.DisableConditionsEvaluator;

/**
 * Class to generate MCC sessions extraction.
 */
public class MCCSessionsExtract extends AbstractExtract implements Initializable
{
    /** The content type EP */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    /** The disable conditions evaluator */
    protected DisableConditionsEvaluator _disableConditionsEvaluator;
    
    private View _extractMCCView;
    private View _extractMCCSessionsView;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _disableConditionsEvaluator = (DisableConditionsEvaluator) manager.lookup(DefaultDisableConditionsEvaluator.ROLE);
    }
    
    public void initialize() throws Exception
    {
        ContentType courseContentType = _contentTypeEP.getExtension(CourseFactory.COURSE_CONTENT_TYPE);
        _extractMCCView = courseContentType.getView("extract-mccsessions");
        _extractMCCSessionsView = courseContentType.getView("extract-mccsessions-sessions");
    }
    
    @Override
    protected String getType(Map<String, String> reportParameters)
    {
        return "mccsessions";
    }
    
    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_CSV);
    }
    
    public void saxProgramItem(ContentHandler handler, String programItemId, Map<String, String> reportParameters)
    {
        // Check the type of the program item (currently, by default, only programs are accepted)
        ProgramItem programItem = _resolver.resolveById(programItemId);
        if (!(programItem instanceof Program program))
        {
            throw new UnsupportedOperationException("The report '" + getType() + "' can be launch only on programs.");
        }
        
        ProgramItemTree programTree = createTreeFromProgramItem(programItem, reportParameters);
        
        try
        {
            handler.startDocument();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType());
            XMLUtils.startElement(handler, "report", attrs);
            
            // Sax column header
            saxHeader(handler);
            
            // Sax the courses from the tree
            saxTree(handler, programTree, new CourseStructure(program, null, null, null, null, null));

            XMLUtils.endElement(handler, "report");
            
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'MCC sessions' extract for program '{}' ({})", program.getTitle(), program.getCode(), e);
        }
    }
    
    /**
     * Add and populate the program item to the {@link ProgramItemTree}
     * @param programItem The program item to add
     * @param reportParameters The report parameters
     * @return a tree of the program
     */
    protected ProgramItemTree createTreeFromProgramItem(ProgramItem programItem, Map<String, String> reportParameters)
    {
        ProgramItemTree programItemTree = new ProgramItemTree(programItem);
        populateTree(programItemTree);
        return programItemTree;
    }
    
    /**
     * Populate the MCC tree.
     * @param tree The MCC tree
     */
    protected void populateTree(ProgramItemTree tree)
    {
        ProgramItem programItem = tree.getCurrent();
        for (ProgramItem child : _odfHelper.getChildProgramItems(programItem))
        {
            ProgramItemTree childTree = tree.addChild(child);
            populateTree(childTree);
        }
    }
    
    /**
     * SAX the MCC tree for the report
     * @param handler The handler
     * @param tree The MCC tree
     * @param courseStructure the course structure
     * @throws SAXException if an error occurs
     */
    protected void saxTree(ContentHandler handler, ProgramItemTree tree, CourseStructure courseStructure) throws SAXException
    {
        ProgramItem programItem = tree.getCurrent();
        
        if (programItem instanceof Course course)
        {
            if (!_pilotageHelper.isExcludedFromMCC(course))
            {
                XMLUtils.startElement(handler, "course");
                
                // Sax the course structure
                courseStructure.saxStructure(handler);
                
                // Sax the course data
                course.dataToSAX(handler, _extractMCCView);
                
                // Sax the course MCC
                saxMCCs(handler, tree);
                
                XMLUtils.endElement(handler, "course");
                
                _saxChildren(handler, tree, programItem, courseStructure);
            }
        }
        else
        {
            _saxChildren(handler, tree, programItem, courseStructure);
        }
    }
    
    private void _saxChildren(ContentHandler handler, ProgramItemTree tree, ProgramItem programItem, CourseStructure courseStructure) throws SAXException
    {
        for (ProgramItemTree child : tree.getChildren())
        {
            saxTree(
                handler, 
                child,
                CourseStructure.of(courseStructure, programItem, _odfHelper)
            );
        }
    }
    
    /**
     * Sax the MCC sessions.
     * @param handler The handler
     * @param tree The MCC tree
     * @throws SAXException If an error occurs
     */
    protected void saxMCCs(ContentHandler handler, ProgramItemTree tree) throws SAXException
    {
        XMLUtils.startElement(handler, "mcc");
        
        // Generate SAX events for MCC sessions
        saxSession(handler, tree, "mccSession1");
        saxSession(handler, tree, "mccSession2");
        
        XMLUtils.endElement(handler, "mcc");
    }
    
    /**
     * Sax the MCC session (1 or 2).
     * @param handler The handler
     * @param tree The MCC tree
     * @param sessionName The session name
     * @throws SAXException If an error occurs
     */
    protected void saxSession(ContentHandler handler, ProgramItemTree tree, String sessionName) throws SAXException
    {
        Course course = (Course) tree.getCurrent();
        
        if (course.hasValue(sessionName) && !_disableConditionsEvaluator.evaluateDisableConditions(_extractMCCSessionsView.getModelViewItem(sessionName).getDefinition(), sessionName, course))
        {
            XMLUtils.startElement(handler, sessionName);
            
            // Get all the full paths from the current educational path, this one can be partial.
            // For example, if have Year1;Semester2, we can obtain:
            //  - Program1;Year1;Semester2
            //  - Program2;SubProgram1;Year1;Semester2
            //  - Program2;SubProgram2;Year1;Semester2
            List<EducationalPath> educationalPaths = _odfHelper.getEducationPathFromPath(tree.getPath().getProgramItems(_resolver));
            
            // For each entry, check if the entry is common or the path correspond to one of the retrieved full educational paths
            for (ModelAwareRepeaterEntry sessionEntry : _odfHelper.getRepeaterEntriesByPath(course.getRepeater(sessionName), educationalPaths).toList())
            {
                _saxSessionEntry(handler, sessionEntry, course);
            }
            
            XMLUtils.endElement(handler, sessionName);
        }
    }
    
    /**
     * Sax a MCC session entry.
     * @param handler The handler
     * @param sessionEntry The session entry name
     * @throws SAXException If an error occurs
     */
    private void _saxSessionEntry(ContentHandler handler, ModelAwareRepeaterEntry sessionEntry, Content rootContent) throws SAXException
    {
        // start entry
        AttributesImpl entryAttrs = new AttributesImpl();
        entryAttrs.addCDATAAttribute("name", String.valueOf(sessionEntry.getPosition()));
        XMLUtils.startElement(handler, "entry", entryAttrs);
        
        String repeaterName = sessionEntry.getHoldingRepeater().getModel().getName();
        sessionEntry.dataToSAX(
            handler,
            (ViewItemAccessor) _extractMCCSessionsView.getModelViewItem(repeaterName),
            RepositoryDataContext.newInstance()
                .withObject(rootContent)
                .withDataPath(repeaterName + "[" + sessionEntry.getPosition() + "]")
        );
        
        // end entry
        XMLUtils.endElement(handler, "entry");
    }
    
    /**
     * Sax the header column for the export
     * @param handler the content handler
     * @throws SAXException if an error occurred
     */
    protected void saxHeader(ContentHandler handler) throws SAXException
    {
        XMLUtils.startElement(handler, "header");
        List<ModelViewItem> modelItem = _extractMCCView.getViewItems()
            .stream()
            .filter(ModelViewItem.class::isInstance)
            .map(ModelViewItem.class::cast)
            .filter(this::_filterModelItem)
            .toList();
        for (ModelViewItem item : modelItem)
        {
            saxColumn(handler, item);
        }
        
        for (ViewItem item : getSessionsViewItems())
        {
            saxColumn(handler, item);
        }
        XMLUtils.endElement(handler, "header");
    }
    
    private boolean _filterModelItem(ModelViewItem viewItem)
    {
        // Do not sax rich text and repeater
        ModelItem definition = viewItem.getDefinition();
        return !(definition instanceof RichTextAttributeDefinition || definition instanceof RepeaterDefinition);
    }
    
    /**
     * Get the merged view item of the MCC sessions repeater
     * @return the list of view item
     */
    protected List<ViewItem> getSessionsViewItems()
    {
        List<ViewItem> items = new ArrayList<>();
        
        ModelViewItemGroup mccSession1View = (ModelViewItemGroup) _extractMCCSessionsView.getViewItem("mccSession1");
        for (ViewItem child : mccSession1View.getViewItems())
        {
            items.add(child);
        }
        
        ModelViewItemGroup mccSession2View = (ModelViewItemGroup) _extractMCCSessionsView.getViewItem("mccSession2");
        for (ViewItem child : mccSession2View.getViewItems())
        {
            if (!mccSession1View.hasModelViewItem(child.getName()))
            {
                items.add(child);
            }
        }
        
        return items;
    }
    
    /**
     * Sax a column from a view item
     * @param handler the content handler
     * @param item the view item
     * @throws SAXException if an error occurred
     */
    protected void saxColumn(ContentHandler handler, ViewItem item) throws SAXException
    {
        if (item instanceof ModelViewItemGroup viewItemGroup)
        {
            for (ViewItem child : viewItemGroup.getViewItems())
            {
                saxColumn(handler, child);
            }
        }
        else if (item instanceof ModelViewItem mViewItem)
        {
            AttributesImpl attrsItem = new AttributesImpl();
            attrsItem.addCDATAAttribute("path", mViewItem.getDefinition().getPath());
            String pathFromSession = _getPathFromSession(mViewItem);
            if (StringUtils.isNotBlank(pathFromSession))
            {
                attrsItem.addCDATAAttribute("pathFromSession", pathFromSession);
            }
            XMLUtils.startElement(handler, "column", attrsItem);
            item.getLabel().toSAX(handler, "label");
            XMLUtils.endElement(handler, "column");
        }
    }
    
    private String _getPathFromSession(ModelViewItem viewItem)
    {
        String path = viewItem.getDefinition().getPath();
        if (StringUtils.startsWithAny(path, "mccSession1/", "mccSession2/"))
        {
            // 12 is the number of character of the mccSessionX/ repeater attribute
            return StringUtils.substring(path, 12);
        }
        
        return null;
    }
    
    private record CourseStructure(Program program, SubProgram subProgram, Container year, Container semester, CourseList level1CourseList, CourseList level2CourseList)
    {
        public static CourseStructure of(CourseStructure structure, ProgramItem item, ODFHelper odfHelper)
        {
            Program localProgram = structure.program();
            SubProgram localSubProgram = structure.subProgram();
            Container localYear = structure.year();
            Container localSemester = structure.semester();
            CourseList localLevel1CourseList = structure.level1CourseList();
            CourseList localLevel2CourseList = structure.level2CourseList();
            
            switch (item)
            {
                case SubProgram subprogram -> localSubProgram = localSubProgram == null ? subprogram : localSubProgram;
                case Container container when odfHelper.isContainerOfTypeYear(container) -> localYear = localYear == null ? container : localYear;
                case Container container -> localSemester = localSemester == null ? container : localSemester;
                case CourseList courseList when courseList.getParentCourses().isEmpty() -> localLevel1CourseList = localLevel1CourseList == null ? courseList : localLevel1CourseList;
                case CourseList courseList -> localLevel2CourseList = localLevel2CourseList == null ? courseList : localLevel2CourseList;
                default -> { /* Do nothing */ }
            }
            
            return new CourseStructure(localProgram, localSubProgram, localYear, localSemester, localLevel1CourseList, localLevel2CourseList);
        }
        
        public void saxStructure(ContentHandler handler) throws SAXException
        {
            XMLUtils.createElement(handler, "programTitle", program.getTitle());
            XMLUtils.createElement(handler, "programCode", program.getDisplayCode());
            
            if (subProgram != null)
            {
                XMLUtils.createElement(handler, "subProgramTitle", subProgram.getTitle());
                XMLUtils.createElement(handler, "subProgramCode", subProgram.getDisplayCode());
            }
            
            if (year != null)
            {
                XMLUtils.createElement(handler, "yearTitle", year.getTitle());
                XMLUtils.createElement(handler, "yearCode", year.getDisplayCode());
            }
            
            if (semester != null)
            {
                XMLUtils.createElement(handler, "semesterTitle", semester.getTitle());
                XMLUtils.createElement(handler, "semesterCode", semester.getDisplayCode());
            }
            
            if (level1CourseList != null)
            {
                XMLUtils.createElement(handler, "level1CourseListTitle", level1CourseList.getTitle());
                XMLUtils.createElement(handler, "level1CourseListCode", level1CourseList.getDisplayCode());
            }
            
            if (level2CourseList != null)
            {
                XMLUtils.createElement(handler, "level2CourseListTitle", level2CourseList.getTitle());
                XMLUtils.createElement(handler, "level2CourseListCode", level2CourseList.getDisplayCode());
            }
        }
    }
}
