/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.helper;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableDefaultContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.odf.workflow.AbstractCreateODFContentFunction;
import org.ametys.odf.workflow.CreateCoursePartFunction;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.model.ModelItem;

import com.opensymphony.workflow.WorkflowException;

/**
 * This component handle the content of the course part tool
 */
public class CoursePartsTreeHelper extends ODFContentsTreeHelper implements Contextualizable
{
    /** The right for course edit */
    public static final String COURSE_EDIT_RIGHT_ID = "ODF_Rights_Course_Edit";

    /** The context */
    protected Context _context;
    /** The right manager */
    protected RightManager _rightManager;
    /** The content workflow helper*/
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
    }
    
    @Override
    protected Map<String, Object> content2Json(Content content, List<String> path)
    {
        Map<String, Object> infos = super.content2Json(content, path);
        
        infos.put("volumesOfHours", _coursePartToJson(content, infos));
        
        return infos;
    }
    
    /**
     * Compute course parts of a course if it hasn't been already computed.
     * @param content The content
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> _coursePartToJson(Content content, Map<String, Object> infos)
    {
        List<String> notEditableDataIndex = (List<String>) infos.computeIfAbsent("notEditableDataIndex", l -> new ArrayList<>());
        
        infos.put("contentId", content.getId());

        Map<String, Object> volumesOfHours = new HashMap<>();
        
        if (content instanceof Course course)
        {
            ModelItem coursePartModelItem = content.getDefinition(Course.CHILD_COURSE_PARTS);
            if (!_contentWorkflowHelper.isAvailableAction(course, 2) || coursePartModelItem instanceof RestrictedModelItem restrictedCoursePart && !restrictedCoursePart.canWrite(course))
            {
                infos.put("notEditableData", true);
            }
            
            List<CoursePart> courseParts = course.getCourseParts();
            
            for (CoursePart coursePart : courseParts)
            {
                String natureId = coursePart.getNature();
                
                if (StringUtils.isNotEmpty(natureId))
                {
                    ModifiableDefaultContent nature = _ametysResolver.resolveById(natureId);
                    
                    // True if we already found one volume of hours for this nature, false otherwise
                    boolean alreadyExists = volumesOfHours.containsKey(natureId);
                    
                    // Check if the CoursePart is editable (edition not available or multiple volume of hours for the nature)
                    if (alreadyExists || !_contentWorkflowHelper.isAvailableAction(coursePart, 2))
                    {
                        notEditableDataIndex.add(natureId);
                    }
                    
                    Map<String, Object> volumeForNature = (Map<String, Object>) volumesOfHours.computeIfAbsent(natureId, __ -> new HashMap<>());
                    
                    // If we have not already found one volume of hours for this nature, add the name to the map
                    if (!alreadyExists)
                    {
                        volumeForNature.put("name", nature.getTitle());
                    }
                    
                    List<Double> values = (List<Double>) volumeForNature.computeIfAbsent("values", __ -> new ArrayList<>());
                    values.add(coursePart.getNumberOfHours());
                }
                else if (getLogger().isWarnEnabled())
                {
                    getLogger().warn("The course part '{}' ({}) does not have a nature, it is mandatory.", coursePart.getTitle(), coursePart.getId());
                }
            }
        }
        // If the content is not a Course, it is not editable
        else
        {
            infos.put("notEditableData", true);
        }
        
        return volumesOfHours;
    }
    
    /**
     * Save edition on the grid
     * @param contentId The id of the content to edit
     * @param changes The changes required
     * @return The result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> saveEdition(String contentId, Map<String, Object> changes)
    {
        Map<String, Object> result = new HashMap<>();
        Map<String, Object> errors = new HashMap<>();
        Course course = _ametysResolver.resolveById(contentId);
        
        if (!_rightManager.currentUserHasRight(COURSE_EDIT_RIGHT_ID, course).equals(RightResult.RIGHT_ALLOW))
        {
            result.put("success", false);
            errors.put("course", Map.of("title", course.getTitle(), "errorMsg", "unauthorized"));
            result.put("errors", errors);
            return result;
        }
        
        for (String key : changes.keySet())
        {
            ModifiableDefaultContent nature = _ametysResolver.resolveById(key);
            
            Optional<CoursePart> optionalCoursePart = course.getCourseParts().stream().filter(coursePartToFilter -> key.equals(coursePartToFilter.getNature())).findFirst();
            
            // If the course already has a CoursePart of the nature, modify it
            if (optionalCoursePart.isPresent() && changes.get(key) != null)
            {
                CoursePart coursePart = optionalCoursePart.get();
                
                if (_contentWorkflowHelper.isAvailableAction(coursePart, 2))
                {
                    try
                    {
                        Map<String, Object> inputs = new HashMap<>();
                        inputs.put(CoursePart.NB_HOURS, changes.get(key));
                        _contentWorkflowHelper.editContent(coursePart, inputs, 2);

                    }
                    catch (AmetysRepositoryException | WorkflowException e)
                    {
                        errors.put(key, Map.of("title", nature.getValue("shortLabel"), "errorMsg", e.getMessage()));
                    }
                    finally
                    {
                        if (course.isLocked())
                        {
                            // The Course has been locked while editing, but since we edit the "CoursePart" it is not unlocked automatically
                            course.unlock();
                        }
                    }
                }
                else
                {
                    errors.put(key, Map.of("title", nature.getValue("shortLabel"), "errorMsg", "unauthorized"));
                }
            }
            // If the course can be modified
            else if (_contentWorkflowHelper.isAvailableAction(course, 2))
            {
                // If there was a coursePart for the nature, and the new value is "null", delete it from the Course
                if (optionalCoursePart.isPresent() && changes.get(key) == null)
                {
                    CoursePart coursePart = optionalCoursePart.get();
                
                    try
                    {
                        List<CoursePart> newCourseParts = course.getCourseParts();
                        // Remove the course part
                        newCourseParts.remove(coursePart);
                        
                        // Edit the course
                        Map<String, Object> inputsForEditCourse = new HashMap<>();
                        inputsForEditCourse.put(Course.CHILD_COURSE_PARTS, newCourseParts);
                        _contentWorkflowHelper.editContent(course, inputsForEditCourse, 2);
                    }
                    catch (AmetysRepositoryException | WorkflowException e)
                    {
                        errors.put(key, Map.of("title", nature.getValue("shortLabel"), "errorMsg", e.getMessage()));
                    }
                }
                // Else, we create the new CoursePart for the Course
                else
                {
                    try
                    {
                        // Create the course part
                        Map<String, Object> inputsForCreation = new HashMap<>();
                        inputsForCreation.put(AbstractCreateODFContentFunction.CONTENT_CATALOG_KEY, course.getCatalog());
                        inputsForCreation.put(CreateCoursePartFunction.COURSE_HOLDER_KEY, course.getId());
                        Map<String, Object> resultForCreation = _contentWorkflowHelper.createContent("course-part", 1, nature.getName(), nature.getTitle(), new String[] {CoursePartFactory.COURSE_PART_CONTENT_TYPE}, new String[0], course.getLanguage(), inputsForCreation);
                        
                        CoursePart coursePart = (CoursePart) resultForCreation.get("org.ametys.cms.repository.Content");
                        
                        // Edit the course part
                        Map<String, Object> inputsForEditCoursePart = new HashMap<>();
                        inputsForEditCoursePart.put(CoursePart.NB_HOURS, changes.get(key));
                        inputsForEditCoursePart.put(CoursePart.NATURE, nature.getId());
                        
                        _contentWorkflowHelper.editContent(coursePart, inputsForEditCoursePart, 2);
                        
                        // Link the course part to the parent course
                        Map<String, Object> inputsForEditCourse = new HashMap<>();
                        List<CoursePart> newCourseParts = course.getCourseParts();
                        newCourseParts.add(coursePart);
                        inputsForEditCourse.put(Course.CHILD_COURSE_PARTS, newCourseParts);
                        
                        _contentWorkflowHelper.editContent(course, inputsForEditCourse, 2);
                    }
                    catch (AmetysRepositoryException | WorkflowException e)
                    {
                        errors.put(key, Map.of("title", nature.getValue("shortLabel"), "errorMsg", e.getMessage()));
                    }
                }
            }
            else
            {
                errors.put(key, Map.of("title", nature.getValue("shortLabel"), "errorMsg", "unauthorized"));
            }
        }
        
        // If there are no errors, then set success to true
        if (errors.isEmpty())
        {
            result.put("success", true);
        }
        else
        {
            // Else, add the errors
            result.put("errors", errors);
            
            // If some changes were successes, set success to true
            if (errors.size() != changes.size())
            {
                result.put("success", true);
            }
        }
        
        return result;
    }
}
