/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rights;

import java.util.ArrayList;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.repository.Content;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.UsersQuery;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper.StepHolderStatus;
import org.ametys.runtime.config.Config;

/**
 * Helper for ODF right
 *
 */
public class ODFRightHelper extends org.ametys.odf.rights.ODFRightHelper
{
    /** The attribute path for MCC manager */
    public static final String MCC_MANAGERS_FIELD_PATH = "odf-mcc-managers";

    /** The pilotage helper */
    protected PilotageHelper _pilotageHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
    }
    
    /**
     * Get the id of profile for MCC managers
     * @return the id of profile for MCC managers
     */
    public String getMCCManagerProfileId()
    {
        return Config.getInstance().getValue("odf.profile.mcc.manager");
    }
    
    /**
     * Get the MCC managers of a {@link ProgramItem} or a {@link OrgUnit}
     * @param content the program item or the orgunit
     * @return the managers or null if not found
     */
    public UserIdentity[] getMCCManagers(Content content)
    {
        if (content instanceof OrgUnit || content instanceof ProgramItem)
        {
            return content.getValue(MCC_MANAGERS_FIELD_PATH);
        }
        
        return null;
    }
    
    /**
     * Build a user query on MCC managers field
     * @param users The users to test.
     * @return the user query
     */
    public Query getMCCManagersQuery(UserIdentity... users)
    {
        return new UsersQuery(MCC_MANAGERS_FIELD_PATH + "_s", Operator.EQ, users);
    }
    
    @Override
    public Query getRolesQuery(UserIdentity user)
    {
        List<Query> userQueries = new ArrayList<>();
        
        userQueries.add(getContributorsQuery(user));
        userQueries.add(getManagersQuery(user));
        userQueries.add(getMCCManagersQuery(user));
        
        return new OrQuery(userQueries);
    }
    
    @Override
    protected List<ProgramItem> computeParentProgramItem(Course course, PermissionContext permissionCtx)
    {
        // Get the container of type "year" that holds this course
        Pair<StepHolderStatus, Container> stepHolder = _pilotageHelper.getStepHolder(course);
        
        switch (stepHolder.getLeft())
        {
            case SINGLE:
                // The course has a unique step holder, returns the nearest ancestors into this step holder to compute parents
                if (permissionCtx.getAncestor() == null)
                {
                    // If no ancestor (=step holder) is defined, set this step holder has ancestor
                    // otherwise compute parents in the defined step holder (current step holder is ignored)
                    permissionCtx.withAncestor(stepHolder.getRight()); // set step holder as ancestor
                }
                return super.computeParentProgramItem(course, permissionCtx);
            case MULTIPLE:
                // The course has several parent containers of type "year"
                if (permissionCtx.getAncestor() != null)
                {
                    // A unique step holder was already found (case of a course under other course(s) : continue to get parents in the tree of this step holder
                    return super.computeParentProgramItem(course, permissionCtx);
                }
                else
                {
                    // cannot determine the unique parent for rights
                    return List.of();
                }
            case NONE:
                // The course has no step holder (ie. none parent container with "year" type)
                if (_odfHelper.getParentPrograms(course).size() == 1)
                {
                    // The course is not shared, rights can be given by its parents
                    return super.computeParentProgramItem(course, permissionCtx);
                }
                else
                {
                    // The course is shared between 2 or more programs: cannot determine the unique parent for rights
                    return List.of();
                }
            case NO_YEAR: // There is no container's nature of type "year", no step holder can be defined
            default:
                return super.computeParentProgramItem(course, permissionCtx);
        }
    }
}
