/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.observations;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.helper.AmetysIdentifiers;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Observer to initialize the code of a thematic.
 */
public class ThematicCreationObserver implements Observer, Serviceable, PluginAware
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    /** The ametys identifiers */
    protected AmetysIdentifiers _identifiers;

    /** The plugin name */
    protected String _pluginName;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _identifiers = (AmetysIdentifiers) manager.lookup(AmetysIdentifiers.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    public boolean supports(Event event)
    {
        if (!RulesManager.isRulesEnabled())
        {
            return false;
        }
        
        if (event.getId().equals(ObservationConstants.EVENT_CONTENT_ADDED))
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            return _contentTypesHelper.isInstanceOf(content, RulesManager.THEMATIC_CONTENT_TYPE);
        }
        
        return false;
    }

    public int getPriority()
    {
        return 0;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        ModifiableContent content = (ModifiableContent) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        
        // Add code
        String code = content.getValue(RulesManager.THEMATIC_CODE);
        if (StringUtils.isEmpty(code))
        {
            content.setValue(RulesManager.THEMATIC_CODE, RulesManager.THEMATICS_PREFIX + _getAndIncrement(RulesManager.THEMATICS_PREFIX));
        }
        
        content.saveChanges();
    }
    
    private long _getAndIncrement(String prefix)
    {
        long counter = _identifiers.readCounter(_pluginName, prefix) + 1;
        _identifiers.saveCounter(_pluginName, prefix, counter);
        return counter;
    }
    
    /**
     * Get or create a node
     * @param parentNode the parent node
     * @param nodeName the name of the node
     * @param nodeType the type of the node
     * @return The retrieved or created node
     * @throws AmetysRepositoryException if an error occurs when manipulating the repository
     */
    protected DefaultTraversableAmetysObject _getOrCreateNode(ModifiableTraversableAmetysObject parentNode, String nodeName, String nodeType) throws AmetysRepositoryException
    {
        DefaultTraversableAmetysObject definitionsNode;
        if (parentNode.hasChild(nodeName))
        {
            definitionsNode = parentNode.getChild(nodeName);
        }
        else
        {
            definitionsNode = parentNode.createChild(nodeName, nodeType);
            parentNode.saveChanges();
        }
        return definitionsNode;
    }
}
