/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.clientsideelement;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.ametys.cms.repository.Content;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.core.ui.Callable;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowAction;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowStep;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Client side element for MCC workflow buttons on a {@link Program}
 */
public class ProgramMCCWorkflowClientSideElement extends MCCWorkflowClientSideElement
{
    /**
     * Get informations on program's state
     * @param programId the id of program
     * @param mccStatus the MCC status
     * @return informations on program's' state
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStatus(String programId, String mccStatus)
    {
        Map<String, Object> results = super.getStatus(List.of(programId));
        
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> allrightContents = (List<Map<String, Object>>) results.get("allright-contents");
        if (allrightContents.isEmpty())
        {
            return results;
        }
        
        Program program = _resolver.resolveById(programId);
        
        Set<Container> childYears = _odfHelper.getYears(program);
        
        if (childYears.isEmpty())
        {
            // Program has no years
            return Map.of("no-years", true);
        }
        
        switch (MCCWorkflowStep.valueOf(mccStatus))
        {
            case MCC_ORGUNIT_VALIDATED:
                if (childYears.stream().allMatch(_mccWorkflowHelper::isMCCOrgUnitValidated))
                {
                    // All child years are orgunit validated
                    return Map.of("all-valid", true);
                }
                else if (!childYears.stream().anyMatch(c -> _mccWorkflowHelper.isAvailableAction(c, MCCWorkflowAction.VALIDATE_ORGUNIT, false))) // check availability of action without checking structure for performance purpose
                {
                    // None child years can be validated
                    return Map.of("no-available", true);
                }
                break;
            case MCC_CFVU_VALIDATED:
                if (childYears.stream().allMatch(_mccWorkflowHelper::isMCCCFVUValidated))
                {
                    // All child years are CFVU validated
                    return Map.of("all-valid", true);
                }
                else if (!childYears.stream().anyMatch(c -> _mccWorkflowHelper.isAvailableAction(c, MCCWorkflowAction.VALIDATE_CFVU, false))) // check availability of action without checking structure for performance purpose
                {
                    // None child years can be validated
                    return Map.of("no-available", true);
                }
                break;
            default:
                throw new IllegalArgumentException("Unexpected value for MCC status : " + mccStatus);
        }
        
        results.put("valid-contents", new ArrayList<>());
        
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> validContents = (List<Map<String, Object>>) results.get("valid-contents");
        for (Container year : childYears)
        {
            getStatus(year, MCCWorkflowStep.valueOf(mccStatus), validContents);
        }
        
        return results;
    }
    
    /**
     * Get status for a year container
     * @param container the container
     * @param mccStatus the MCC status
     * @param validContents The list of valid contents
     */
    protected void getStatus(Container container, MCCWorkflowStep mccStatus, List<Map<String, Object>> validContents)
    {
        Map<String, Object> defaultContentParams = getContentDefaultParameters(container);
        
        if (_mccWorkflowHelper.isStepValid(container, mccStatus))
        {
            Map<String, Object> validContentParams = new HashMap<>(defaultContentParams);
            validContentParams.put("description", _getValidDescription(container));
            validContents.add(validContentParams);
        }
    }
    
    /**
     * Get i18n description when the MCC workflow status is active
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getActiveDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("active-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
    
    /**
     * Get i18n description when the MCC workflow status match the current step of the content
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getValidDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("valid-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
    
    /**
     * Get minimun date for MCC orgunit validation
     * @param programItemId the id of root program
     * @return the min date
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public LocalDate getMinDateForMCCOrgUnitValidation(String programItemId)
    {
        TraversableProgramPart programItem = _resolver.resolveById(programItemId);
        
        List<String> childYearIds = _odfHelper.getYears(programItem)
                .stream()
                .filter(c -> _mccWorkflowHelper.isAvailableAction(c, MCCWorkflowAction.VALIDATE_ORGUNIT))
                .map(Container::getId)
                .toList();
        
        return super.getMinDateForMCCOrgUnitValidation(childYearIds);
    }
    
    /**
     * Get minimun date for MCC CFVU validation
     * @param programItemId the id of root program
     * @return the min date
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public LocalDate getMinDateForMCCCFVUValidation(String programItemId)
    {
        TraversableProgramPart programItem = _resolver.resolveById(programItemId);
        
        List<String> childYearIds = _odfHelper.getYears(programItem)
                .stream()
                .filter(c -> _mccWorkflowHelper.isAvailableAction(c, MCCWorkflowAction.VALIDATE_CFVU))
                .map(Container::getId)
                .toList();
        
        return super.getMinDateForMCCCFVUValidation(childYearIds);
    }
    
    /**
     * Validate the MCC to orgunit level for all child years of a program
     * @param programItemId the id of root program
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @return the result
     */
    @Callable (rights = "ODF_Pilotage_Global_MCC_Orgunit_Validated_Rights", paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> validateOrgunitMCC(String programItemId, String validationDate, String comment)
    {
        TraversableProgramPart programItem = _resolver.resolveById(programItemId);
        
        List<String> childYearIds = _odfHelper.getYears(programItem)
                .stream()
                .filter(y -> !_mccWorkflowHelper.isMCCOrgUnitValidated(y)) // ignore years that are already validated
                .map(Container::getId)
                .toList();
        
        // Do not check rights on years
        return validateOrgunitMCC(childYearIds, null, validationDate, comment, Map.of());
    }
    
    /**
     * Validate the MCC to CFVU level for all child years of a program
     * @param programItemId the id of root program
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = "ODF_Pilotage_Global_CFVU_MCC_Validated_Rights", paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public Map<String, Object> validateMCCForCVFU(String programItemId, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        TraversableProgramPart programItem = _resolver.resolveById(programItemId);
        
        List<String> containerIds = _odfHelper.getYears(programItem)
                .stream()
                .filter(y -> !_mccWorkflowHelper.isMCCCFVUValidated(y)) // ignore years that are already validated
                .map(Container::getId)
                .toList();
        
        // Do not check rights on years
        return validateMCCForCFVU(containerIds, null, validationDate, comment, contextualParameters);
        
    }
}
