/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.content.compare.ContentComparator;
import org.ametys.cms.content.compare.ContentComparatorResult;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.report.impl.tree.ProgramItemTree;
import org.ametys.plugins.odfpilotage.report.impl.tree.ProgramItemTree.ChangeType;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.model.View;

/**
 * Class to generate maquette diff catalog report (based on maquette report)..
 */
public class MaquetteDiffExtract extends AbstractMaquetteExtract
{
    /** The key for the old catalog */
    public static final String PARAMETER_CATALOG_OLD = "catalogOld";

    /** The maquette report view for courses comparison */
    protected View _courseCompareView;
    
    private ContentComparator _contentComparator;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentComparator = (ContentComparator) manager.lookup(ContentComparator.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        super.initialize();
        _courseCompareView = getView(CourseFactory.COURSE_CONTENT_TYPE, "report-maquette-compare");
    }
    
    @Override
    protected String getType(Map<String, String> reportParameters)
    {
        return "maquettediff";
    }
    
    @Override
    protected boolean isGeneric()
    {
        return false;
    }
    
    @Override
    protected boolean isCompatibleSchedulable(AbstractReportSchedulable schedulable)
    {
        return schedulable.getId().equals("org.ametys.plugins.odfpilotage.schedulable.OrgUnitDiffReportSchedulable");
    }
    
    @Override
    protected ProgramItemTree createTreeFromProgramItem(ProgramItem programItem, Map<String, String> reportParameters)
    {
        ProgramItem oldProgramItem = _findOldProgramItem(programItem, reportParameters.get(PARAMETER_CATALOG_OLD));
        ContentComparatorResult changes = _getObjectChanges(oldProgramItem, programItem);
        ProgramItemTree programItemTree = new ProgramItemTree(programItem, changes);
        populateTree(programItemTree);
        return programItemTree;
    }
    
    @Override
    protected void populateTree(ProgramItemTree tree)
    {
        ProgramItem oldProgramItem = (ProgramItem) tree.getChange().getContent1();
        ProgramItem newProgramItem = (ProgramItem) tree.getChange().getContent2();
        
        if (oldProgramItem == null)
        {
            // Only new
            List<ProgramItem> children = getOrderedChildren(newProgramItem);
            for (ProgramItem child : children)
            {
                ContentComparatorResult changes = _getObjectChanges(null, child);
                ProgramItemTree childTree = tree.addChild(child, changes);
                populateTree(childTree);
            }
        }
        else if (newProgramItem == null)
        {
            // Only old
            List<ProgramItem> children = getOrderedChildren(oldProgramItem);
            for (ProgramItem child : children)
            {
                ContentComparatorResult changes = _getObjectChanges(child, null);
                ProgramItemTree childTree = tree.addChild(child, changes);
                populateTree(childTree);
            }
        }
        else
        {
            List<ProgramItem> oldChildren = getOrderedChildren(oldProgramItem);
            List<ProgramItem> newChildren = getOrderedChildren(newProgramItem);

            // Add new program items
            for (ProgramItem newChild : newChildren)
            {
                ProgramItem currentOldChild = null;
                Class<? extends ProgramItem> newChildClass = newChild.getClass();
                String newChildCode = newChild.getCode();
                for (ProgramItem oldChild : oldChildren)
                {
                    if (oldChild.getCode().equals(newChildCode) && oldChild.getClass().equals(newChildClass))
                    {
                        currentOldChild = oldChild;
                        // Decrease the size of old children to do the next operations faster
                        oldChildren.remove(oldChild);
                        break;
                    }
                }

                ContentComparatorResult changes = _getObjectChanges(currentOldChild, newChild);
                ProgramItemTree childTree = tree.addChild(newChild, changes);
                populateTree(childTree);
            }
            
            // Add old program items
            for (ProgramItem oldChild : oldChildren)
            {
                ContentComparatorResult changes = _getObjectChanges(oldChild, null);
                ProgramItemTree childTree = tree.addChild(oldChild, changes);
                populateTree(childTree);
            }
        }
    }
    
    @Override
    protected void saxContent(ContentHandler handler, String tagName, Content content, View view, ProgramItemTree tree, boolean saxHierarchy) throws SAXException
    {
        // Sax the old version if needed (MODIFIED_OLD)
        ContentComparatorResult changes = tree.getChange();
        if (!changes.areEquals() && changes.getContent1() != null && changes.getContent2() != null)
        {
            super.saxContent(handler, tagName, changes.getContent1(), view, tree, false);
        }
        
        super.saxContent(handler, tagName, content, view, tree, true);
    }
    
    @Override
    protected AttributesImpl getProgramItemAttributes(ProgramItem programItem, ProgramItemTree tree)
    {
        AttributesImpl attr = super.getProgramItemAttributes(programItem, tree);
        
        ContentComparatorResult changes = tree.getChange();
        if (!changes.areEquals())
        {
            ChangeType changeType;
            if (changes.getContent1() == null)
            {
                changeType = ChangeType.ADDED;
            }
            else if (changes.getContent2() == null)
            {
                changeType = ChangeType.REMOVED;
            }
            else if (changes.getContent1().equals(programItem))
            {
                changeType = ChangeType.MODIFIED_OLD;
            }
            else
            {
                changeType = ChangeType.MODIFIED_NEW;
            }
            attr.addCDATAAttribute("changeType", changeType.name().toLowerCase());
        }
        
        return attr;
    }
    
    private ContentComparatorResult _getObjectChanges(ProgramItem oldProgramItem, ProgramItem newProgramItem)
    {
        Content oldContent = (Content) oldProgramItem;
        Content newContent = (Content) newProgramItem;
        
        ContentComparatorResult changes = new ContentComparatorResult(oldContent, newContent);
        
        // Only if old and new program items are both there, tests some attributes
        if (changes.areEquals() && !_equalsProgramItem(oldContent, newContent))
        {
            changes.setNotEquals();
        }
        
        return changes;
    }
    
    private boolean _equalsProgramItem(Content oldProgramItem, Content newProgramItem)
    {
        try
        {
            ContentComparatorResult programItemChanges = _contentComparator.compare(oldProgramItem, newProgramItem, _courseCompareView);
            
            // Additional tests for Course
            if (programItemChanges.areEquals() && oldProgramItem instanceof Course oldCourse && newProgramItem instanceof Course newCourse)
            {
                return _equalsCourse(oldCourse, newCourse);
            }
            
            return programItemChanges.areEquals();
        }
        catch (AmetysRepositoryException e)
        {
            getLogger().error("Une erreur est survenue pour l'ELP {} - {}", oldProgramItem.getValue(ProgramItem.CODE), newProgramItem.getTitle(), e);
        }
        
        return true;
    }
    
    private boolean _equalsCourse(Course oldCourse, Course newCourse)
    {
        return Objects.equals(oldCourse.getParentCourseLists().size(), newCourse.getParentCourseLists().size())
                && Objects.equals(_getContainerEtape(oldCourse), _getContainerEtape(newCourse))
                && Objects.equals(_countNbHoursByNature(oldCourse), _countNbHoursByNature(newCourse));
    }
    
    /**
     * Get course's etape porteuse if exist
     * @param course the current course
     * @return course's etape porteuse if exist, else return null
     */
    private Container _getContainerEtape(Course course)
    {
        return Optional.ofNullable((ContentValue) course.getValue("etapePorteuse"))
                .flatMap(ContentValue::getContentIfExists)
                .map(Container.class::cast)
                .orElse(null);
    }
    
    private Map<String, Double> _countNbHoursByNature(Course course)
    {
        Map<String, Double> nbHours = new HashMap<>();
        for (CoursePart coursePart : course.getCourseParts())
        {
            String nature = Optional.of(coursePart.<ContentValue>getValue("nature"))
                    .map(ContentValue::getContentId)
                    .orElse(StringUtils.EMPTY);
            nbHours.put(nature, nbHours.getOrDefault(nature, 0d) + coursePart.getNumberOfHours());
        }
        return nbHours;
    }
    
    @Override
    protected void saxGlobalInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        super.saxGlobalInformations(handler, programItem, reportParameters);
        
        String oldCatalog = reportParameters.get(PARAMETER_CATALOG_OLD);
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", oldCatalog);
        XMLUtils.createElement(handler, "catalogOld", attrs, _catalogsManager.getCatalog(oldCatalog).getTitle());
    }
    
    /**
     * Find the equivalent content in the new catalog
     * @param <T> Type of content to find
     * @param content Content in the current catalog
     * @return New equivalent content or null
     */
    private <T extends ProgramItem> T _findOldProgramItem(T content, String oldCatalog)
    {
        return _odfHelper.getProgramItem(content, oldCatalog, content.getLanguage());
    }
}
