/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.schedulable;

import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.SchedulerException;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.data.Binary;
import org.ametys.cms.schedule.AbstractSendingMailSchedulable;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.odf.program.Container;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.WorkflowException;

/**
 * {@link Schedulable} for MCC validated PDF generation
 */
public class MCCValidatedPDFSchedulable extends AbstractSendingMailSchedulable
{
    /** The schedulable ID */
    public static final String ID = MCCValidatedPDFSchedulable.class.getName();
    
    private static String _CONTAINER_IDS_SUCCESS_KEY = "containerIdsInSuccess";
    private static String _CONTAINER_IDS_ERROR_KEY = "containerIdsInError";
    
    /** The avalon source resolver. */
    protected SourceResolver _sourceResolver;
    
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;
    
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _mccWorkflowHelper = (MCCWorkflowHelper) manager.lookup(MCCWorkflowHelper.ROLE);
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public void _doExecute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        @SuppressWarnings("unchecked")
        List<String> containerIds = (List<String>) jobDataMap.get(Scheduler.PARAM_VALUES_PREFIX + "containerIds");
        LocalDate date = (LocalDate) jobDataMap.get(Scheduler.PARAM_VALUES_PREFIX + "date");
        
        List<String> containerIdsInSuccess = new ArrayList<>();
        List<String> containerIdsInError = new ArrayList<>();
        
        for (String containerId : containerIds)
        {
            Source pdfSource = null;
            try
            {
                Container container = _resolver.resolveById(containerId);
                String encodeURI = _getPDFURI(container, date);
                pdfSource = _sourceResolver.resolveURI(encodeURI);
                
                Binary binary = _getPDFAsBinary(pdfSource, container, date);
                _registerPDF(container, binary, date);
                
                containerIdsInSuccess.add(containerId);
            }
            catch (Exception e)
            {
                containerIdsInError.add(containerId);
                getLogger().error("An error occurred generating the MCC validated PDF for container with id '{}'", containerId, e);
            }
            finally
            {
                _sourceResolver.release(pdfSource);
            }
        }
        
        context.put(_CONTAINER_IDS_SUCCESS_KEY, containerIdsInSuccess);
        context.put(_CONTAINER_IDS_ERROR_KEY, containerIdsInError);
        if (!containerIdsInError.isEmpty())
        {
            List<Container> containersInError = containerIdsInError.stream()
                .map(id -> _resolver.resolveById(id))
                .filter(Container.class::isInstance)
                .map(Container.class::cast)
                .toList();
            
            for (Container container : containersInError)
            {
                _mccWorkflowHelper.invalidateMCCForCVFU(container, null, false, null, false);
            }
            
            throw new SchedulerException("Some containers have not been MCC validated for CFVU because an error occurred during the PDF generation. For more information, see logs above.");
        }
    }
    
    /**
     * Get the URI to generate the PDF for a given container
     * @param container the container
     * @param date the validation date
     * @return the PDF URI
     */
    protected String _getPDFURI(Container container, LocalDate date)
    {
        Map<String, String> parameters = new HashMap<>();
        parameters.put("programItem", container.getId());
        parameters.put("reportId", "org.ametys.plugins.odfpilotage.report.MCCReport");
        parameters.put("isDefinitive", "true");
        return RulesManager.isRulesEnabled()
                ? URIUtils.encodeURI("cocoon://_plugins/odf-pilotage/final/container-rules.pdf", parameters)
                : URIUtils.encodeURI("cocoon://_plugins/odf-pilotage/pilotage/report.pdf", parameters);
    }

    /**
     * Get the MCC validated PDF as binary
     * @param pdfSource the PDF source
     * @param container the container
     * @param date the validation date
     * @return the binary
     * @throws IOException if an error occurred
     */
    protected Binary _getPDFAsBinary(Source pdfSource, Container container, LocalDate date) throws IOException
    {
        Binary binary = new Binary();
        try (InputStream is = pdfSource.getInputStream())
        {
            binary.setEncoding("UTF-8");
            binary.setMimeType("application/pdf");
            binary.setLastModificationDate(ZonedDateTime.now());
            binary.setFilename(_getPDFFileName(container, date));
            binary.setInputStream(is);
        }
        return binary;
    }
    
    private String _getPDFFileName(Container container, LocalDate date)
    {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        String dateAsString = date.format(formatter);
        return "MCC " + container.getTitle() + " " + container.getCode() + " " + dateAsString + ".pdf";
    }
    
    /**
     * Register the MCC validated PDF to the container
     * @param container the container
     * @param binary PDF as binary
     * @param date the validation date
     * @throws WorkflowException if an error occurred
     */
    protected void _registerPDF(Container container, Binary binary, LocalDate date) throws WorkflowException
    {
        ModifiableModelAwareRepeater repeater = container.getRepeater("mcc-validated-pdf", true);
        ModifiableModelAwareRepeaterEntry entry = repeater.addEntry();
        
        entry.setValue("pdf", binary);
        entry.setValue("date", date);
        
        Map<String, Object> paramsEdit = new HashMap<>();
        paramsEdit.put(CheckRightsCondition.FORCE, true); // Ignore the right condition
        paramsEdit.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, new HashMap<>());
        
        _contentWorkflowHelper.doAction(container, MCCWorkflowHelper.MCC_WORKFLOW_ACTION_ID, paramsEdit);
    }

    @Override
    protected boolean _isMailBodyInHTML(JobExecutionContext context) throws Exception
    {
        return true;
    }
    
    @Override
    protected I18nizableText _getSuccessMailSubject(JobExecutionContext context) throws Exception
    {
        return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_CFVU_MAIL_SUBJECT");
    }

    @Override
    @SuppressWarnings("unchecked")
    protected String _getSuccessMailBody(JobExecutionContext context, String language) throws Exception
    {
        MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withTitle(_getSuccessMailSubject(context));
        
        String contextualParametersAsString = context.getJobDetail().getJobDataMap().getString(Scheduler.PARAM_VALUES_PREFIX + "contextualParameters");
        Map<String, Object> contextualParameters = _jsonUtils.convertJsonToMap(contextualParametersAsString);
        
        List<String> containerIdsInSuccess = (List<String>) context.get(_CONTAINER_IDS_SUCCESS_KEY);
        if (!containerIdsInSuccess.isEmpty())
        {
            bodyBuilder.addMessage(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_CFVU_MAIL_MSG"));
            bodyBuilder.addMessage(_contentsToHtml(containerIdsInSuccess, contextualParameters));
        }
        
        bodyBuilder.withLanguage(language);
        
        return bodyBuilder.build();
    }
    
    @Override
    protected I18nizableText _getErrorMailSubject(JobExecutionContext context) throws Exception
    {
        return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_CFVU_MAIL_SUBJECT");
    }

    @Override
    @SuppressWarnings("unchecked")
    protected String _getErrorMailBody(JobExecutionContext context, String language, Throwable throwable) throws Exception
    {
        MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withTitle(_getSuccessMailSubject(context));
        
        String contextualParametersAsString = context.getJobDetail().getJobDataMap().getString(Scheduler.PARAM_VALUES_PREFIX + "contextualParameters");
        Map<String, Object> contextualParameters = _jsonUtils.convertJsonToMap(contextualParametersAsString);
        
        List<String> containerIdsInSuccess = (List<String>) context.get(_CONTAINER_IDS_SUCCESS_KEY);
        if (!containerIdsInSuccess.isEmpty())
        {
            bodyBuilder.addMessage(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_CFVU_MAIL_MSG"));
            bodyBuilder.addMessage(_contentsToHtml(containerIdsInSuccess, contextualParameters));
        }
        
        List<String> containerIdsInError = (List<String>) context.get(_CONTAINER_IDS_ERROR_KEY);
        if (!containerIdsInError.isEmpty())
        {
            bodyBuilder.addMessage(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_CFVU_MAIL_ERROR_MSG"));
            bodyBuilder.addMessage(_contentsToHtml(containerIdsInError, contextualParameters));
            bodyBuilder.addMessage(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_CFVU_MAIL_ERROR_LOG_MSG"));
        }
        
        bodyBuilder.withLanguage(language);
        
        return bodyBuilder.build();
    }
    
    private String _contentsToHtml(List<String> contentIds, Map<String, Object> contextualParameters)
    {
        StringBuilder htmlMessage = new StringBuilder("<ul>");
        for (String id : contentIds)
        {
            Container container = _resolver.resolveById(id);
            
            String href = _contentHelper.getContentBOUrl(container, contextualParameters);
            htmlMessage.append("<li>");
            htmlMessage.append("<a href=\"").append(href).append("\">").append(container.getTitle()).append("</a>");
            htmlMessage.append("</li>");
        }
        htmlMessage.append("</ul>");
        
        return htmlMessage.toString();
    }
}
