/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.tree;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.Content;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.tree.AbstractStaticODFTreeIndicator;
import org.ametys.odf.tree.ODFTreeIndicator;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * {@link ODFTreeIndicator} to indicates the {@link Course} or the {@link CourseList} are excluded from the MCC
 */
public class ExcludedFromMCCIndicator extends AbstractStaticODFTreeIndicator
{
    /** The pilotage helper */
    protected PilotageHelper _pilotageHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _pilotageHelper = (PilotageHelper) smanager.lookup(PilotageHelper.ROLE);
    }
    
    public IndicatorData getIndicatorData(Content content)
    {
        if (content instanceof Course course)
        {
            if (_pilotageHelper.isExcludedFromMCC(course))
            {
                return new IndicatorData(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_EXCLUDED_FROM_MCC_TREE_INDICATORS_TOOLTIP"), null, getIconGlyph(), Map.of());
            }
            else if (_pilotageHelper.isParentExcludedInMCC(course))
            {
                return new IndicatorData(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_EXCLUDED_FROM_MCC_TREE_INDICATORS_PARENT_TOOLTIP"), null, "grey-color " + getIconGlyph(), Map.of());
            }
        }
        else if (content instanceof CourseList courseList)
        {
            if (_pilotageHelper.isParentExcludedInMCC(courseList))
            {
                return new IndicatorData(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_EXCLUDED_FROM_MCC_TREE_INDICATORS_PARENT_TOOLTIP"), null, "grey-color " + getIconGlyph(), Map.of());
            }
        }
        
        return null;
    }
}
