/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.validator;

import java.util.Arrays;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.codec.binary.StringUtils;

import org.ametys.cms.contenttype.AttributeDefinition;
import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.holder.group.IndexableRepeater;
import org.ametys.cms.data.holder.group.IndexableRepeaterEntry;
import org.ametys.cms.model.ContentRestrictedRepeaterDefinition;
import org.ametys.cms.repository.Content;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.disableconditions.DefaultDisableConditionsEvaluator;
import org.ametys.runtime.model.disableconditions.DisableConditionsEvaluator;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * Validates that the mcc regime is compatible with "modalite" in the mccSessions
 */
public class CourseMccRegimeValidator extends AbstractContentValidator implements Serviceable
{
    private ContentTypesHelper _contentTypesHelper;
    private DisableConditionsEvaluator _disableConditionsEvaluator;
    private PilotageHelper _pilotageHelper;

    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _disableConditionsEvaluator = (DisableConditionsEvaluator) manager.lookup(DefaultDisableConditionsEvaluator.ROLE);
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
    }
    
    public ValidationResult validate(Content content)
    {
        ValidationResult result = new ValidationResult();
        
        AttributeDefinition mccRegimeDefinition = (AttributeDefinition) content.getDefinition(PilotageHelper.COURSE_MCC_REGIME);
        if (!_disableConditionsEvaluator.evaluateDisableConditions(mccRegimeDefinition, PilotageHelper.COURSE_MCC_REGIME, content))
        {
            ContentValue mccRegime = content.getValue(PilotageHelper.COURSE_MCC_REGIME);
            if (mccRegime != null && mccRegime.getContentId() != null)
            {
                Content mccRegimeContent = mccRegime.getContentIfExists().orElse(null);
                if (mccRegimeContent != null)
                {
                    result.addResult(_check(mccRegimeContent.getId(), content, "mccSession1"));
                    result.addResult(_check(mccRegimeContent.getId(), content, "mccSession2"));
                }
            }
        }
        
        return result;
    }

    private ValidationResult _check(String mccRegimeId, Content content, String mccSession)
    {
        ValidationResult result = new ValidationResult();
        
        ContentRestrictedRepeaterDefinition mccSessionDefinition = (ContentRestrictedRepeaterDefinition) content.getDefinition(mccSession);
        if (!_disableConditionsEvaluator.evaluateDisableConditions(mccSessionDefinition, mccSession, content)
                && _pilotageHelper.canWriteMccRestrictions(content, mccSession))
        {
            IndexableRepeater repeater = content.getRepeater(mccSession);
            if (repeater != null)
            {
                int index = 0;
                for (IndexableRepeaterEntry entry : repeater.getEntries())
                {
                    index++;
                    
                    ContentValue modalite = entry.getValue("modalite");
                    if (modalite != null && modalite.getContentId() != null)
                    {
                        ContentAttributeDefinition mccSessionModaliteDefinition = (ContentAttributeDefinition) content.getDefinition(mccSession + "/modalite");
                        
                        Content modaliteContent = modalite.getContentIfExists().orElse(null);
                        if (modaliteContent != null)
                        {
                            if (!_contentTypesHelper.isInstanceOf(modaliteContent, mccSessionModaliteDefinition.getContentTypeId()))
                            {
                                // Not a valid contenttype
                                result.addError(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_COURSE_MCCREGIME_VALIDATOR_ERROR_1", Map.of(
                                        "fieldSession", content.getDefinition(mccSession).getLabel(),
                                        "fieldSessionPosition", new I18nizableText(Integer.toString(index)),
                                        "fieldModalite", mccSessionModaliteDefinition.getLabel()
                                )));
                            }
                            else
                            {
                                ContentValue[] regimes = modaliteContent.getValue("compatibleRegimes");
                                if (regimes != null && regimes.length > 0)
                                {
                                    if (!Arrays.stream(regimes).map(ContentValue::getContentId).anyMatch(c -> StringUtils.equals(mccRegimeId, c)))
                                    {
                                        // Incompatible
                                        result.addError(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_COURSE_MCCREGIME_VALIDATOR_ERROR_2", Map.of(
                                                "fieldSession", content.getDefinition(mccSession).getLabel(),
                                                "fieldSessionPosition", new I18nizableText(Integer.toString(index)),
                                                "fieldModalite", mccSessionModaliteDefinition.getLabel()
                                        )));
                                    }
                                }
                            }
                        }
                        // The value stored does not exist (anymore?), let's validate it can be empty
                        else if (mccSessionModaliteDefinition.getValidator() != null)
                        {
                            ValidationResult localResult = mccSessionModaliteDefinition.getValidator().validate(null);
                            if (localResult.hasErrors())
                            {
                                // Not existing (anymore?)
                                result.addError(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_COURSE_MCCREGIME_VALIDATOR_ERROR_3", Map.of(
                                        "fieldSession", content.getDefinition(mccSession).getLabel(),
                                        "fieldSessionPosition", new I18nizableText(Integer.toString(index)),
                                        "fieldModalite", mccSessionModaliteDefinition.getLabel()
                                )));
                            }
                        }
                    }
                }
            }
        }
        
        return result;
    }

    public ValidationResult validate(Content content, Map<String, Object> values, View view)
    {
        // Nothing because there are already widgets for classic edition, and we want to let other editions ends even if it leads to invalid content
        return ValidationResult.empty();
    }
}
