/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.validator;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.holder.group.IndexableRepeater;
import org.ametys.cms.data.holder.group.IndexableRepeaterEntry;
import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.Program;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.disableconditions.DefaultDisableConditionsEvaluator;
import org.ametys.runtime.model.disableconditions.DisableConditionsEvaluator;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * Checks that the mcc skills are compatible with the course path in the mccSessions
 */
public class CourseMccSkillsValidator extends AbstractContentValidator implements Serviceable, Initializable
{
    private static final String __MCC_SESSION1 = "mccSession1";
    private static final String __MCC_SESSION2 = "mccSession2";
    
    private static final String __MCC_SESSION_COMMON = "common";
    private static final String __MCC_SESSION_PATH = "path";
    private static final String __MCC_SESSION_NOTES = "notes";
    private static final String __MCC_SESSION_NOTES_SKILLS = "skills";
    private static final String __MCC_SESSION_NOTES_SKILLS_SKILL = "skill";
    private static final String __MCC_SESSION_NOTES_SKILLS_PATH = "path";
    
    private static final String __SKILLS_BY_PROGRAMS_CACHE_ID = CourseMccSkillsValidator.class.getName() + "$skillsByPrograms";
    
    private DisableConditionsEvaluator _disableConditionsEvaluator;
    private ODFHelper _odfHelper;
    private ODFSkillsHelper _odfSkillsHelper;
    private AmetysObjectResolver _resolver;
    private AbstractCacheManager _cacheManager;
    private boolean _isSkillsEnabled;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _disableConditionsEvaluator = (DisableConditionsEvaluator) manager.lookup(DefaultDisableConditionsEvaluator.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _odfSkillsHelper = (ODFSkillsHelper) manager.lookup(ODFSkillsHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
    }
    
    @Override
    public void initialize() throws ConfigurationException
    {
        super.initialize();
        
        _isSkillsEnabled = Config.getInstance().getValue("odf.skills.enabled");
        
        if (!_cacheManager.hasCache(__SKILLS_BY_PROGRAMS_CACHE_ID))
        {
            _cacheManager.createRequestCache(__SKILLS_BY_PROGRAMS_CACHE_ID,
                    new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_SKILLS_BY_PROGRAMS_LABEL"),
                    new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_SKILLS_BY_PROGRAMS_DESCRIPTION"),
                    true);
        }
    }
    
    /**
     * Get the microskill associated to a program. This method manage a cache.
     * @param programId The program identifier
     * @return <code>null</code> if the identifier is unknown or not a program, otherwise skill identifiers
     */
    private Set<String> _getProgramMicroSkills(String programId)
    {
        Cache<String, Set<String>> cache = _cacheManager.get(__SKILLS_BY_PROGRAMS_CACHE_ID);
        return cache.get(programId, this::_populateCacheEntry);
    }
    
    private Set<String> _populateCacheEntry(String id)
    {
        try
        {
            Content content = _resolver.resolveById(id);
            if (content instanceof Program program)
            {
                return _odfSkillsHelper.getProgramMicroSkills(program).collect(Collectors.toSet());
            }
        }
        catch (AmetysRepositoryException e)
        {
            // Silently ignore: the program does not exists anymore or it is not a program
        }
        
        return null;
    }
    
    public ValidationResult validate(Content content)
    {
        ValidationResult result = new ValidationResult();
        
        if (_isSkillsEnabled && content instanceof Course course)
        {
            // Get the educational path of the course if it is not shared
            Optional<EducationalPath> courseEducationalPath = _getEducationalPath(course);
            
            // Check the unicity of the skills under each note by educational path and check that every skills is attached to the right program
            result.addResult(_checkSession(course, __MCC_SESSION1, courseEducationalPath));
            result.addResult(_checkSession(course, __MCC_SESSION2, courseEducationalPath));
        }
        
        return result;
    }
    
    private ValidationResult _checkSession(Course course, String mccSession, Optional<EducationalPath> courseEducationalPath)
    {
        ValidationResult result = new ValidationResult();
        
        RepeaterDefinition mccSessionDefinition = (RepeaterDefinition) course.getDefinition(mccSession);
        
        // Check if the mcc session is not disabled
        if (!_disableConditionsEvaluator.evaluateDisableConditions(mccSessionDefinition, mccSession, course))
        {
            // Build common i18n parameters
            RepeaterDefinition noteDefinition = (RepeaterDefinition) mccSessionDefinition.getModelItem(__MCC_SESSION_NOTES);
            Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
            i18nParams.put("sessionRepeater", mccSessionDefinition.getLabel());
            i18nParams.put("noteRepeater", noteDefinition.getLabel());
            i18nParams.put("skillRepeater", noteDefinition.getModelItem(__MCC_SESSION_NOTES_SKILLS).getLabel());
            
            // Process each mcc session entry
            Optional.ofNullable(course.getRepeater(mccSession))
                .map(IndexableRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .map(entry -> _checkSessionEntry(entry, courseEducationalPath, i18nParams))
                .forEach(result::addResult);
        }
        
        return result;
    }
    
    private ValidationResult _checkSessionEntry(IndexableRepeaterEntry sessionEntry, Optional<EducationalPath> courseEducationalPath, Map<String, I18nizableTextParameter> commonI18nParams)
    {
        ValidationResult result = new ValidationResult();
        
        // Build common i18n parameters
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>(commonI18nParams);
        i18nParams.put("sessionPosition", new I18nizableText(Integer.toString(sessionEntry.getPosition())));
        
        // Get the educational path of the entry if it is not common and if the course educational path is empty
        Optional<EducationalPath> sessionEntryEducationalPath = courseEducationalPath
            .or(() ->
                Optional.of(sessionEntry)
                    // Don't get the educational path for common entries
                    .filter(entry -> !entry.getValue(__MCC_SESSION_COMMON, true, true))
                    .map(entry -> entry.getValue(__MCC_SESSION_PATH))
            );
        
        // Process each note entry
        Optional.ofNullable(sessionEntry.getRepeater(__MCC_SESSION_NOTES))
            .map(IndexableRepeater::getEntries)
            .map(List::stream)
            .orElseGet(Stream::of)
            .map(noteEntry -> _checkNoteEntry(noteEntry, sessionEntryEducationalPath, i18nParams))
            .forEach(result::addResult);
        
        return result;
    }
    
    private ValidationResult _checkNoteEntry(IndexableRepeaterEntry noteEntry, Optional<EducationalPath> sessionEntryEducationalPath, Map<String, I18nizableTextParameter> commonI18nParams)
    {
        ValidationResult result = new ValidationResult();
        
        // Build common i18n parameters
        Map<String, I18nizableTextParameter> noteI18nParams = new HashMap<>(commonI18nParams);
        noteI18nParams.put("notePosition", new I18nizableText(Integer.toString(noteEntry.getPosition())));

        // Map used to detect duplicated skill for each educational path in the note entry
        Map<EducationalPath, Set<ContentValue>> skillsByPath = new HashMap<>();
        
        List<? extends IndexableRepeaterEntry> skillEntries = Optional.ofNullable(noteEntry.getRepeater(__MCC_SESSION_NOTES_SKILLS))
            .map(IndexableRepeater::getEntries)
            .map(List::stream)
            .orElseGet(Stream::of)
            .toList();
        
        // Process each skill entry
        for (IndexableRepeaterEntry skillEntry : skillEntries)
        {
            ContentValue skill = skillEntry.getValue(__MCC_SESSION_NOTES_SKILLS_SKILL);
            EducationalPath educationalPath = sessionEntryEducationalPath.orElseGet(() -> skillEntry.getValue(__MCC_SESSION_NOTES_SKILLS_PATH));
            
            // If one of this data is null, the mandatory validator should react
            if (skill != null && educationalPath != null)
            {
                // Get the skills already present for the current educational path
                Set<ContentValue> skillsForPath = skillsByPath.computeIfAbsent(educationalPath, __ -> new HashSet<>());
                
                // The skill is already defined for this educational path in this note entry
                if (!skillsForPath.add(skill))
                {
                    // Error because duplication
                    result.addError(_buildDuplicationMessage(noteI18nParams, skill, educationalPath));
                }
                // Size of educational path can be zero if the course has no parent program
                else if (educationalPath.getProgramItemIds().size() > 0)
                {
                    // Check if the skill belong to the program of the educational path
                    
                    String programId = educationalPath.getProgramItemIds().get(0);
                    Set<String> programMicroSkills = _getProgramMicroSkills(programId);

                    // The program micro skills may be null if the first element of the educational path is not a program or not exists
                    if (programMicroSkills != null && !programMicroSkills.contains(skill.getContentId()))
                    {
                        // If there are skills that are not in the program, add the error
                        result.addError(_buildInconsistentMessage(noteI18nParams, skillEntry.getPosition(), skill, programId));
                    }
                }
            }
        }
        
        return result;
    }
    
    public ValidationResult validate(Content content, Map<String, Object> values, View view)
    {
        ValidationResult result = new ValidationResult();
        
        /* FIXME ODF-4031 Need the current edition context to filter errors, modification in table should not display errors from another program
        // Check if the received values contains some data for mccSession[1|2]/notes/skills
        // Otherwise, it is useless to run this validator
        if (_isSkillsEnabled && content instanceof Course course && _hasSkillsInNotes(values))
        {
            // Get the educational path of the course if it is not shared
            Optional<EducationalPath> courseEducationalPath = _getEducationalPath(course);
            
            // Check the unicity of the skills under each note by educational path and check that every skills is attached to the right program
            result.addResult(_checkSessionFromValues(course, values, __MCC_SESSION1, courseEducationalPath));
            result.addResult(_checkSessionFromValues(course, values, __MCC_SESSION2, courseEducationalPath));
        }
        */
        
        return result;
    }
    
    private Optional<EducationalPath> _getEducationalPath(Course course)
    {
        return Optional.of(course)
            // Course non shared (only one educational path)
            .filter(c -> !c.<Boolean>getValue(ProgramItem.SHARED_PROPERTY))
            // Get the only parent programs (it can be zero)
            .map(_odfHelper::getParentPrograms)
            .filter(parents -> parents.size() == 1)
            .map(Set::stream)
            .flatMap(Stream::findFirst)
            // Convert it as EducationalPath
            .map(EducationalPath::of);
    }
    
    private I18nizableText _buildDuplicationMessage(Map<String, I18nizableTextParameter> commonI18nParams, ContentValue skill, EducationalPath educationalPath)
    {
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>(commonI18nParams);
        i18nParams.put("skill", new I18nizableText(skill.getContentIfExists().map(Content::getTitle).orElseGet(() -> skill.getContentId())));
        i18nParams.put("educationalPath", new I18nizableText(_odfHelper.getEducationalPathAsString(educationalPath)));
        return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_COURSE_MCCSKILLS_VALIDATOR_ERROR_DUPLICATED", i18nParams);
    }
    
    private I18nizableText _buildInconsistentMessage(Map<String, I18nizableTextParameter> commonI18nParams, int skillPosition, ContentValue skill, String programId)
    {
        Map<String, I18nizableTextParameter> i18nParams = new HashMap<>(commonI18nParams);
        i18nParams.put("skillPosition", new I18nizableText(Integer.toString(skillPosition)));
        i18nParams.put("skill", new I18nizableText(skill.getContentIfExists().map(Content::getTitle).orElseGet(() -> skill.getContentId())));
        i18nParams.put("program", new I18nizableText(_resolver.<Program>resolveById(programId).getTitle()));
        return new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_COURSE_MCCSKILLS_VALIDATOR_ERROR_INCONSISTENT", i18nParams);
    }
}
