/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.odfpilotage;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;

/**
 * Action get the last MCC validated PDF from given attributes name and value
 */
public class GetMCCValidatedAction extends ServiceableAction
{
    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The catalog manager */
    protected CatalogsManager _catalogsManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _mccWorkflowHelper = (MCCWorkflowHelper) smanager.lookup(MCCWorkflowHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _catalogsManager = (CatalogsManager) smanager.lookup(CatalogsManager.ROLE);
    }

    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String lang = Optional.ofNullable(request.getParameter("lang")).orElseGet(() -> Config.getInstance().getValue("odf.programs.lang"));
        String catalog = Optional.ofNullable(request.getParameter("catalog")).orElseGet(() -> _catalogsManager.getDefaultCatalogName());

        Map<String, String> attributeParameters = _getAttributeParameters(request);
        List<Container> contents = _getYearFromAttributeParameters(lang, catalog, attributeParameters);
        int size = contents.size();
        switch (size)
        {
            case 0 :
                // throw 404, no content has been found
                getLogger().error("No year content has been found in catalog '" + catalog + "' and language '" + lang + "' with attributes " + _attributes2String(attributeParameters));
                return null;
            case 1 :
                Container content = contents.get(0);
                ModelAwareRepeaterEntry entry = _mccWorkflowHelper.getLastMCCValidatedEntry(content);
                if (entry != null)
                {
                    // PDF has been found, return it
                    return Map.of("id", content.getId(), "path", "mcc-validated-pdf[" + entry.getPosition() + "]/pdf");
                }
                else
                {
                    // throw 404, no MCC validated PDF has been found
                    getLogger().error("No validated MCC PDF has been found in catalog '" + catalog + "' and language '" + lang + "' with attributes " + _attributes2String(attributeParameters));
                    return null;
                }
            default :
                // throw 404 for multiple content found
                getLogger().error(size + " year contents have been found in catalog '" + catalog + "' and language '" + lang + "' with attributes " + _attributes2String(attributeParameters));
                return null;
        }
    }
    
    private String _attributes2String(Map<String, String> attributeParameters)
    {
        return attributeParameters.entrySet()
            .stream()
            .map(e -> e.getKey() + " '" + e.getValue() + "'")
            .collect(Collectors.joining(" and "));
    }
    
    private Map<String, String> _getAttributeParameters(Request request)
    {
        Map<String, String> attributesValues = new HashMap<>();
        
        int i = 1;
        String attrName = request.getParameter("attr" + i);
        while (attrName != null)
        {
            String attrValue = request.getParameter("value" + i);
            attributesValues.put(attrName, attrValue);
            i++;
            attrName = request.getParameter("attr" + i);
        }
        
        return attributesValues;
    }
    
    private List<Container> _getYearFromAttributeParameters(String lang, String catalog, Map<String, String> attributeParameters)
    {
        Optional<String> yearId = _odfHelper.getYearId();
        if (yearId.isPresent())
        {
            AndExpression expr = new AndExpression();
            expr.add(new ContentTypeExpression(Operator.EQ, ContainerFactory.CONTAINER_CONTENT_TYPE));
            expr.add(new LanguageExpression(Operator.EQ, lang));
            expr.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
            expr.add(new StringExpression(Container.NATURE, Operator.EQ, yearId.get()));
            
            attributeParameters.entrySet()
                .stream()
                .map(entry -> new StringExpression(entry.getKey(), Operator.EQ, entry.getValue()))
                .forEach(expr::add);
            
            String xpathQuery = ContentQueryHelper.getContentXPathQuery(expr);
            AmetysObjectIterable<Container> contents = _resolver.query(xpathQuery);
            return contents.stream().toList();
        }
        
        return List.of();
    }
}
