/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.xslt;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;

import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.XMLUtils;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.ametys.core.util.DateUtils;
import org.ametys.core.util.dom.AmetysNodeList;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.model.RepositoryDataContext;

/**
 * MCC helper component to be used from XSL stylesheets.
 */
public class MCCXSLTHelper implements Serviceable, LogEnabled
{
    /** The Ametys resolver */
    protected static AmetysObjectResolver _ametysObjectResolver;
    
    /** The MCC workflow helper */
    protected static MCCWorkflowHelper _mccWorkflowHelper;
    
    /** The logger */
    protected static Logger _logger;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _ametysObjectResolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _mccWorkflowHelper = (MCCWorkflowHelper) smanager.lookup(MCCWorkflowHelper.ROLE);
    }
    
    @Override
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }
    
    /**
     * Get the more recent PDF for validated MCC for a {@link Container}
     * @param containerId the container id
     * @return the pdf as an ametys node list
     */
    public static AmetysNodeList getMCCValidatedPDF(String containerId)
    {
        try
        {
            Container container = _ametysObjectResolver.resolveById(containerId);
            ModelAwareRepeaterEntry entry = _mccWorkflowHelper.getLastMCCValidatedEntry(container);
            if (entry != null)
            {
                SAXTransformerFactory saxTransformerFactory = (SAXTransformerFactory) TransformerFactory.newInstance();
                TransformerHandler th = saxTransformerFactory.newTransformerHandler();
                
                DOMResult result = new DOMResult();
                th.setResult(result);
                
                th.startDocument();
                XMLUtils.startElement(th, "value");
                if (entry.hasValue("pdf"))
                {
                    entry.dataToSAX(
                        th,
                        "pdf",
                        RepositoryDataContext.newInstance()
                            .withDataPath("mcc-validated-pdf[" + entry.getPosition() + "]/pdf")
                    );
                }
                XMLUtils.endElement(th, "value");
                th.endDocument();
                
                List<Node> values = new ArrayList<>();
                
                // #getChildNodes() returns a NodeList that contains the value(s) saxed
                // we cannot returns directly this NodeList because saxed values should be wrapped into a <value> tag.
                NodeList childNodes = result.getNode().getFirstChild().getChildNodes();
                for (int i = 0; i < childNodes.getLength(); i++)
                {
                    Node n = childNodes.item(i);
                    values.add(n);
                }
                
                return new AmetysNodeList(values);
            }
        }
        catch (Exception e)
        {
            _logger.error("An error occurred getting the MCC validated PDF", e);
        }
        
        return null;
    }
    
    /**
     * Get the MCC CFVU validation date
     * @param containerId the container id
     * @return the MCC CFVU validation date
     */
    public static String getMCCCFVUValidationDate(String containerId)
    {
        Container container = _ametysObjectResolver.resolveById(containerId);
        LocalDate validationDate = _mccWorkflowHelper.getMCCCFVUValidationDate(container);
        return DateUtils.localDateToString(validationDate);
    }
}
