/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.clientsideelement;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.quartz.SchedulerException;

import org.ametys.cms.clientsideelement.SmartContentClientSideElement;
import org.ametys.cms.repository.Content;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.schedule.Runnable;
import org.ametys.core.schedule.Runnable.FireProcess;
import org.ametys.core.schedule.Runnable.MisfirePolicy;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.SchedulableExtensionPoint;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.program.Container;
import org.ametys.plugins.core.impl.schedule.DefaultRunnable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowException;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowAction;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.schedulable.MCCValidatedPDFSchedulable;
import org.ametys.plugins.repository.lock.LockHelper;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Client side element for MCC workflow actions button on a {@link Container}
 */
public class MCCWorkflowClientSideElement extends SmartContentClientSideElement
{
    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;

    /** The ODF helper */
    protected ODFHelper _odfHelper;

    /** The scheduler */
    protected Scheduler _scheduler;
    
    /** The scheduler extention point */
    protected SchedulableExtensionPoint _schedulerEP;
    
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _mccWorkflowHelper = (MCCWorkflowHelper) manager.lookup(MCCWorkflowHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _scheduler = (Scheduler) manager.lookup(Scheduler.ROLE);
        _schedulerEP = (SchedulableExtensionPoint) manager.lookup(SchedulableExtensionPoint.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        List<Script> scripts = super.getScripts(ignoreRights, contextParameters);
        if (!scripts.isEmpty())
        {
            String mccWorkflowActionName = (String) scripts.get(0).getParameters().get("mcc-workflow-action");
            if (StringUtils.isNotEmpty(mccWorkflowActionName))
            {
                MCCWorkflowAction mccWorkflowAction = MCCWorkflowAction.valueOf(mccWorkflowActionName);
                if (MCCWorkflowAction.VALIDATE_RULES.equals(mccWorkflowAction) || MCCWorkflowAction.REFUSE_RULES_ORGUNIT_CONTROL.equals(mccWorkflowAction))
                {
                    if (!RulesManager.isRulesEnabled())
                    {
                        return List.of();
                    }
                }
            }
        }
        return scripts;
    }
    
    /**
     * Get informations on contents' state
     * @param containerIds the ids of container
     * @param mccWorkflowAction the MCC action
     * @return informations on contents' state
     */
    @SuppressWarnings("unchecked")
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStatus(List<String> containerIds, String mccWorkflowAction)
    {
        Map<String, Object> results = super.getStatus(containerIds);
        
        results.put("active-contents", new ArrayList<>());
        results.put("invalidmccstatus-contents", new ArrayList<>());
        results.put("noyear-contents", new ArrayList<>());
        
        List<Map<String, Object>> invalidContents = (List<Map<String, Object>>) results.get("invalidmccstatus-contents");
        List<Map<String, Object>> noYearContents = (List<Map<String, Object>>) results.get("noyear-contents");
        List<Map<String, Object>> allrightContents = (List<Map<String, Object>>) results.get("allright-contents");
        
        for (Map<String, Object> allRightContent : allrightContents)
        {
            String containerId = (String) allRightContent.get("id");
            Container container = _resolver.resolveById(containerId);
            
            _getStatus(container, MCCWorkflowAction.valueOf(mccWorkflowAction), noYearContents, invalidContents);
        }
        
        List<Object> invalidContentIds = invalidContents.stream()
            .map(p -> p.get("id"))
            .toList();
        
        List<Object> noYearContentIds = noYearContents.stream()
                .map(p -> p.get("id"))
                .toList();
        
        
        results.put("allright-contents", allrightContents.stream().filter(p -> !invalidContentIds.contains(p.get("id")) && !noYearContentIds.contains(p.get("id"))).toList());
        
        return results;
    }
    
    /**
     * Get status for a container
     * @param container the container
     * @param workflowAction the MCC workflow action
     * @param noYearContents The list of no-year containers
     * @param invalidMCCStatusContents the list of container with invalid MCC status for the current action
     */
    protected void _getStatus(Container container, MCCWorkflowAction workflowAction, List<Map<String, Object>> noYearContents, List<Map<String, Object>> invalidMCCStatusContents)
    {
        Map<String, Object> defaultContentParams = getContentDefaultParameters(container);
        
        if (!_odfHelper.isContainerOfTypeYear(container))
        {
            Map<String, Object> noYearContentParams = new HashMap<>(defaultContentParams);
            noYearContentParams.put("description", _getNoYearDescription(container));
            noYearContents.add(noYearContentParams);
            return;
        }
        
        boolean disabled = !_mccWorkflowHelper.isAvailableAction(container, workflowAction);
        if (disabled)
        {
            Map<String, Object> invalidContentParams = new HashMap<>(defaultContentParams);
            invalidContentParams.put("description", _getInvalidMCCStatusDescription(container));
            invalidMCCStatusContents.add(invalidContentParams);
        }
    }
    
    /**
     * Get i18n description when the MCC workflow status does not allow to do action
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getInvalidMCCStatusDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("invalidmccstatus-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
    
    /**
     * Get i18n description when the container is not of type year
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getNoYearDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("noyear-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
    
    /**
     * Validate the rules
     * @param containerIds the container ids
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> validateRules(List<String> containerIds, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate localDate  = StringUtils.isNotEmpty(validationDate) ? DateUtils.parseLocalDate(validationDate) : LocalDate.now();
        
        return doMCCAction(containerIds, "ODF_Pilotage_MCC_Rules_Validated_Rights", container -> _mccWorkflowHelper.validateRules(container, localDate, user, comment));
    }
    
    /**
     * Refuse control of rules to orgunit level
     * @param containerIds the container ids
     * @param sendMail true to notify by mail the author of precede step
     * @param comment The comment of email
     * @param reset true to reset MCC workflow
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> refuseRulesOrgunitControl(List<String> containerIds, boolean sendMail, String comment, boolean reset)
    {
        UserIdentity user = _currentUserProvider.getUser();
        return doMCCAction(containerIds, "ODF_Pilotage_MCC_Rules_Refuse_Control_Rights", container -> _mccWorkflowHelper.refuseRulesOrgUnitControl(container, user, sendMail, comment, reset));
    }
    
    /**
     * Validate the rules
     * @param containerIds the container ids
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> validateMCC(List<String> containerIds, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate localDate  = StringUtils.isNotEmpty(validationDate) ? DateUtils.parseLocalDate(validationDate) : LocalDate.now();
        
        return doMCCAction(containerIds, "ODF_Pilotage_MCC_Validated_Rights", container -> _mccWorkflowHelper.validateMCC(container, localDate, user, comment));
    }
    
    /**
     * Refuse control of MCC to orgunit level
     * @param containerIds the container ids
     * @param sendMail true to notify by mail the author of precede step
     * @param comment The comment of email
     * @param reset true to reset MCC workflow
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> refuseMCCOrgunitControl(List<String> containerIds, boolean sendMail, String comment, boolean reset)
    {
        UserIdentity user = _currentUserProvider.getUser();
        return doMCCAction(containerIds, "ODF_Pilotage_MCC_Validated_Rights", container -> _mccWorkflowHelper.refuseMCCOrgunitControl(container, user, sendMail, comment, reset));
    }
    
    /**
     * Get minimun date for MCC orgunit validation for given containers
     * @param containerIds the container ids
     * @return the minimun date
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public LocalDate getMinDateForMCCOrgUnitControl(List<String> containerIds)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate minDate = null;
        for (String containerId : containerIds)
        {
            Container container = _resolver.resolveById(containerId);
            
            if (_rightManager.hasRight(user, "ODF_Pilotage_MCC_Orgunit_Control_Rights", container) == RightResult.RIGHT_ALLOW)
            {
                LocalDate cMinDate = _mccWorkflowHelper.getMinDateForMCCOrgUnitControl(container);
                if (minDate == null || cMinDate.isAfter(minDate))
                {
                    minDate = cMinDate;
                }
            }
        }
        
        return minDate;
    }
    
    /**
     * Get minimun date for MCC orgunit validation for given containers
     * @param containerIds the container ids
     * @return the minimun date
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public LocalDate getMinDateForMCCOrgUnitValidation(List<String> containerIds)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate minDate = null;
        for (String containerId : containerIds)
        {
            Container container = _resolver.resolveById(containerId);
            
            if (_rightManager.hasRight(user, "ODF_Pilotage_MCC_Orgunit_Validated_Rights", container) == RightResult.RIGHT_ALLOW)
            {
                LocalDate cMinDate = _mccWorkflowHelper.getMinDateForMCCOrgUnitValidation(container);
                if (minDate == null || cMinDate.isAfter(minDate))
                {
                    minDate = cMinDate;
                }
            }
        }
        
        return minDate;
    }
    
    /**
     * Control MCC to orgunit level
     * @param containerIds the container ids
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> controlOrgunitMCC(List<String> containerIds, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate localDate  = StringUtils.isNotEmpty(validationDate) ? DateUtils.parseLocalDate(validationDate) : LocalDate.now();
        
        return doMCCAction(containerIds, "ODF_Pilotage_MCC_Orgunit_Control_Rights", container -> _mccWorkflowHelper.controlMCCForOrgunit(container, localDate, user, comment));
    }
    
    /**
     * Validate the MCC to orgunit level
     * @param containerIds the container ids
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> validateOrgunitMCC(List<String> containerIds, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        return validateOrgunitMCC(containerIds, "ODF_Pilotage_MCC_Orgunit_Validated_Rights", validationDate, comment, contextualParameters);
    }
    
    /**
     * Validate the MCC to orgunit level
     * @param containerIds the container ids
     * @param rightId The right id to check. Can be null to skip right checking.
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    public Map<String, Object> validateOrgunitMCC(List<String> containerIds, String rightId, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate localDate  = StringUtils.isNotEmpty(validationDate) ? DateUtils.parseLocalDate(validationDate) : LocalDate.now();
        
        return doMCCAction(containerIds, rightId, container -> _mccWorkflowHelper.validateMCCForOrgunit(container, localDate, user, comment));
    }
    
    /**
     * Refuse CFVU control of MCC
     * @param containerIds the container ids
     * @param sendMail true to notify by mail the author of precede step
     * @param comment The comment of email
     * @param reset true to reset MCC workflow
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> refuseCFVUControl(List<String> containerIds, boolean sendMail, String comment, boolean reset)
    {
        UserIdentity user = _currentUserProvider.getUser();
        return doMCCAction(containerIds, "ODF_Pilotage_MCC_CFVU_Refuse_Control_Rights", container -> _mccWorkflowHelper.refuseCFVUControl(container, user, sendMail, comment, reset));
    }
    
    /**
     * Control MCC to CFVU level
     * @param containerIds the container ids
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> controlMCCForCFVU(List<String> containerIds, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate localDate  = StringUtils.isNotEmpty(validationDate) ? DateUtils.parseLocalDate(validationDate) : LocalDate.now();
        
        return doMCCAction(containerIds, "ODF_Pilotage_CFVU_MCC_Control_Rights", container -> _mccWorkflowHelper.controlMCCForCVFU(container, localDate, user, comment));
    }
    
    /**
     * Refuse validation of MCC to CFVU level
     * @param containerIds the container ids
     * @param sendMail true to notify by mail the author of precede step
     * @param comment The comment of email
     * @param reset true to reset MCC workflow
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> refuseCFVUValidation(List<String> containerIds, boolean sendMail, String comment, boolean reset)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        return doMCCAction(containerIds, "ODF_Pilotage_MCC_CFVU_Refuse_Validation_Rights", container -> _mccWorkflowHelper.refuseCFVUValidation(container, user, sendMail, comment, reset));
    }
    
    /**
     * Get minimun date for MCC CFVU control for given containers
     * @param containerIds the container ids
     * @return the minimun date for MCC CFVU control
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public LocalDate getMinDateForMCCCFVUControl(List<String> containerIds)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate minDate = null;
        for (String containerId : containerIds)
        {
            Container container = _resolver.resolveById(containerId);
            
            if (_rightManager.hasRight(user, "ODF_Pilotage_CFVU_MCC_Control_Rights", container) == RightResult.RIGHT_ALLOW)
            {
                LocalDate cMinDate = _mccWorkflowHelper.getMinDateForMCCCFVUControl(container);
                if (minDate == null || cMinDate.isAfter(minDate))
                {
                    minDate = cMinDate;
                }
            }
        }
        
        return minDate;
    }
    
    /**
     * Refuse validation of MCC to orgunit level
     * @param containerIds the container ids
     * @param sendMail true to notify by mail the author of precede step
     * @param comment The comment of email
     * @param reset true to reset MCC workflow
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> refuseOrgUnitValidation(List<String> containerIds, boolean sendMail, String comment, boolean reset)
    {
        UserIdentity user = _currentUserProvider.getUser();
        return doMCCAction(containerIds, "ODF_Pilotage_MCC_Orgunit_Refuse_Validation_Right", container -> _mccWorkflowHelper.refuseOrgunitValidation(container, user, sendMail, comment, reset));
    }
    
    /**
     * Get minimun date for MCC CFVU validation for given container
     * @param containerIds the container ids
     * @return the minimun date for MCC CFVU validation
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public LocalDate getMinDateForMCCCFVUValidation(List<String> containerIds)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate minDate = null;
        for (String containerId : containerIds)
        {
            Container container = _resolver.resolveById(containerId);
            
            if (_rightManager.hasRight(user, "ODF_Pilotage_CFVU_MCC_Validated_Rights", container) == RightResult.RIGHT_ALLOW)
            {
                LocalDate cMinDate = _mccWorkflowHelper.getMinDateForMCCCFVUValidation(container);
                if (minDate == null || cMinDate.isAfter(minDate))
                {
                    minDate = cMinDate;
                }
            }
        }
        
        return minDate;
        
    }
    
    /**
     * Validate the MCC
     * @param containerIds the container ids
     * @param validationDate the date of validation. Can be empty or null.
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> validateMCCForCFVU(List<String> containerIds, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        return validateMCCForCFVU(containerIds, "ODF_Pilotage_CFVU_MCC_Validated_Rights", validationDate, comment, contextualParameters);
    }
    
    /**
     * Validate the MCC
     * @param containerIds the container ids
     * @param rightId The right id to check. Can be null to skip right checking.
     * @param validationDate the date of validation. Can be empty or null.
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    public Map<String, Object> validateMCCForCFVU(List<String> containerIds, String rightId, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        LocalDate localDate  = StringUtils.isNotEmpty(validationDate) ? DateUtils.parseLocalDate(validationDate) : LocalDate.now();
        
        Map<String, Object> results = doMCCAction(containerIds, rightId, container -> _mccWorkflowHelper.validateMCCForCVFU(container, localDate, user, comment));
        
        try
        {
            @SuppressWarnings("unchecked")
            List<String> successContentIds = (List<String>) results.get("success-contents");
            if (!successContentIds.isEmpty())
            {
                Map<String, Object> parameters = new HashMap<>();
                parameters.put("date", localDate);
                parameters.put("containerIds", successContentIds);
                parameters.put("contextualParameters", _jsonUtils.convertObjectToJson(contextualParameters));
                
                Schedulable mccValidatedPDFSchedulable = _schedulerEP.getExtension(MCCValidatedPDFSchedulable.ID);
                String runnableId = mccValidatedPDFSchedulable.getId() + "$" + UUID.randomUUID();
                Runnable runnable = new DefaultRunnable(runnableId,
                        mccValidatedPDFSchedulable.getLabel(),
                        mccValidatedPDFSchedulable.getDescription(),
                        FireProcess.NOW,
                        null /* cron*/,
                        mccValidatedPDFSchedulable.getId(),
                        true /* removable */,
                        false /* modifiable */,
                        false /* deactivatable */,
                        MisfirePolicy.FIRE_ONCE,
                        true /* isVolatile */,
                        _currentUserProvider.getUser(),
                        parameters
                    );
                    
                _scheduler.scheduleJob(runnable);
                results.put("schedulable-id", runnableId);
                results.put("schedulable-label", mccValidatedPDFSchedulable.getLabel());
            }
        }
        catch (SchedulerException e)
        {
            results.put("error", "schedulable-error");
        }
        
        return results;
    }
    
    /**
     * Invalidate the MCC
     * @param containerIds the container ids
     * @param sendMail true to notify by mail the author of precede step
     * @param comment The comment of email
     * @param reset true to reset MCC workflow
     * @return the result
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> invalidateMCCForCFVU(List<String> containerIds, boolean sendMail, String comment, boolean reset)
    {
        UserIdentity user = _currentUserProvider.getUser();
        return doMCCAction(containerIds, "ODF_Pilotage_MCC_CFVU_Revert_Validation_Right", container -> _mccWorkflowHelper.invalidateMCCForCVFU(container, user, sendMail, comment, reset));
    }
    
    /**
     * Do MCC action
     * @param containerIds the container ids
     * @param rightId the right id to check. Can be null to skip right checking
     * @param mccWorkflowAction The MCC workflow action to apply
     * @return the results
     */
    protected Map<String, Object> doMCCAction(List<String> containerIds, String rightId, Function<Container, Boolean> mccWorkflowAction)
    {
        Map<String, Object> results = new HashMap<>();
        
        List<Map<String, Object>> noRightContents = new ArrayList<>();
        List<Map<String, Object>> lockedContents = new ArrayList<>();
        List<Map<String, Object>> workflowErrorContents = new ArrayList<>();
        List<String> successContents = new ArrayList<>();
        
        UserIdentity user = _currentUserProvider.getUser();
        
        for (String containerId : containerIds)
        {
            Container container = _resolver.resolveById(containerId);
            
            try
            {
                if (rightId != null && _rightManager.hasRight(user, rightId, container) != RightResult.RIGHT_ALLOW)
                {
                    noRightContents.add(Map.of("id", containerId, "title", container.getTitle()));
                }
                else if (container.isLocked() && !LockHelper.isLockOwner(container, user))
                {
                    lockedContents.add(Map.of("id", containerId, "title", container.getTitle()));
                }
                else
                {
                    mccWorkflowAction.apply(container);
                    successContents.add(containerId);
                }
            }
            catch (MCCWorkflowException e)
            {
                workflowErrorContents.add(Map.of("id", containerId, "title", container.getTitle(), "errorType", e.getType() != null ? e.getType().name() : null));
            }
        }
        
        results.put("success-contents", successContents);
        results.put("noright-contents", noRightContents);
        results.put("locked-contents", lockedContents);
        results.put("invalidmccstatus-contents", workflowErrorContents);
        
        return results;
        
    }
}
