/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.cost.entity;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Object representing cost computation informations for an item (can be a program item or a court part).
 */
public class ProgramItemData
{
    private Effectives _effectives;
    private VolumesOfHours _volumesOfHours;
    private EqTD _eqTD;
    private Double _weight;
    private Set<String> _pathes;
    
    // Only for course parts
    private Groups _groups;
    private NormDetails _normDetails;
    
    private boolean _isCoursePart;

    /**
     * Constructor
     */
    public ProgramItemData()
    {
        _effectives = null;
        _volumesOfHours = null;
        _groups = null;
        _eqTD = null;
        _weight = null;
        _normDetails = null;
        _pathes = null;
        _isCoursePart = false;
    }
    
    /**
     * Get the effectives
     * @return the effectives
     */
    public Effectives getEffectives()
    {
        return _effectives;
    }
    
    /**
     * Set the effectives
     * @param effectives the effectives
     */
    public void setEffectives(Effectives effectives)
    {
        _effectives = effectives;
    }
    
    /**
     * Get the volumes of hours
     * @return the volumes of hours
     */
    public VolumesOfHours getVolumesOfHours()
    {
        return _volumesOfHours;
    }
    
    /**
     * Set the volumes of hours
     * @param volumesOfHours the volumes of hours
     */
    public void setVolumesOfHours(VolumesOfHours volumesOfHours)
    {
        _volumesOfHours = volumesOfHours;
    }

    /**
     * Get the groups
     * @return the groups
     */
    public Groups getGroups()
    {
        return _groups;
    }
    
    /**
     * Set the groups
     * @param groups the groups
     */
    public void setGroups(Groups groups)
    {
        _groups = groups;
    }
    
    /**
     * Get the td equivalent
     * @return the td equivalent
     */
    public EqTD getEqTD()
    {
        return _eqTD;
    }
    
    /**
     * Set the td equivalent
     * @param eqTD the td equivalent
     */
    public void setEqTD(EqTD eqTD)
    {
        _eqTD = eqTD;
    }
    
    /**
     * Get the weight
     * @return the weight
     */
    public Double getWeight()
    {
        return _weight;
    }
    
    /**
     * Set the weight
     * @param weight the weight
     */
    public void setWeight(Double weight)
    {
        _weight = weight;
    }
    
    /**
     * Get all the H/E ratios by item path, a ratio can be empty if effectives are empty.
     * @return a map of H/E ratios
     */
    public Map<String, Optional<Double>> getHeRatios()
    {
        return Optional.ofNullable(_effectives)
            .map(Effectives::getLocalEffectiveByPath)
            .map(Map::keySet)
            .map(Collection::stream)
            .orElseGet(Stream::of)
            .collect(Collectors.toMap(Function.identity(), this::getHeRatio));
    }
    
    /**
     * Get the H/E ratio
     * @param programItemPath the program item path
     * @return the H/E ratio
     */
    public Optional<Double> getHeRatio(String programItemPath)
    {
        Optional<Double> localEffective = Optional.ofNullable(_effectives)
                .map(eff -> eff.getLocalEffective(programItemPath))
                .filter(eff -> eff > 0);
        
        if (localEffective.isEmpty())
        {
            return Optional.empty();
        }

        Double proratedEqTD = Optional.ofNullable(_eqTD)
                .map(eqTD -> eqTD.getProratedEqTD(programItemPath))
                .orElse(0D);

        return Optional.of(proratedEqTD / localEffective.get());
    }
    
    /**
     * Get the possible pathes of the element
     * @return the possible pathes
     */
    public Set<String> getPathes()
    {
        return _pathes;
    }
    
    /**
     * Set the possible pathes of the element
     * @param pathes the possible pathes
     */
    public void setPathes(Set<String> pathes)
    {
        _pathes = pathes;
    }
    
    /**
     * Get the norm details
     * @return the norm details
     */
    public NormDetails getNormDetails()
    {
        return _normDetails;
    }
    
    /**
     * Set the norm details
     * @param normDetails the norm details
     */
    public void setNormDetails(NormDetails normDetails)
    {
        _normDetails = normDetails;
    }
    
    /**
     * <code>true</code> if the current data concern a course part, otherwise <code>false</code>.
     * @return <code>true</code> if the current data concern a course part
     */
    public boolean isCoursePart()
    {
        return _isCoursePart;
    }

    /**
     * Set <code>true</code> to indicate that the current data concern a course part
     * @param isCoursePart the course part status
     */
    public void setIsCoursePart(boolean isCoursePart)
    {
        _isCoursePart = isCoursePart;
    }
}
