/*
 * Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.helper;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.cms.search.ui.model.SearchUIModel;
import org.ametys.core.ui.Callable;
import org.ametys.odf.EducationalPathHelper;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.data.type.EducationalPathRepositoryElementType;
import org.ametys.odf.program.Program;
import org.ametys.odf.rights.ODFRightHelper;
import org.ametys.odf.workflow.EditContextualizedDataFunction;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.type.DataContext;

/**
 * Helper for MCC course tree grid: a contentsWithView that also send specific info such as the educationalPath
 */
public class MCCCourseTreeGridHelper extends org.ametys.plugins.odfpilotage.helper.ContentsWithViewTreeGridHelper
{
    private ODFHelper _odfHelper;
    private EducationalPathHelper _educationalPathHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _educationalPathHelper = (EducationalPathHelper) smanager.lookup(EducationalPathHelper.ROLE);
    }
    
    @Override
    protected Map<String, Object> content2Json(Content content, List<String> path)
    {
        Map<String, Object> content2Json = super.content2Json(content, path);

        String localClientPath = path.stream().collect(Collectors.joining(";"));
        
        if (content instanceof ProgramItem programItem)
        {
            // Add educational path (can only be done on server side since it can be partial on client side)
            List<EducationalPath> educationalPaths = _odfHelper.getEducationalPaths(programItem);
            List<Map<String, String>> educationalPathsJson = educationalPaths.stream()
                .map(ep -> Pair.of(ep.toString(), ep))
                .filter(pep -> pep.getKey().endsWith(localClientPath))
                .map(pep -> Pair.of(pep.getKey(), _odfHelper.getEducationalPathAsString(pep.getValue())))
                .map(pep -> Map.of("path", pep.getKey(), "label", pep.getValue()))
                .toList();

            content2Json.put("educationalPaths", educationalPathsJson);
        }
        
        return content2Json;
    }
    
    @Override
    protected void fillContentData(Content content, SearchUIModel searchUIModel, List<String> path, Map<String, Object> content2Json)
    {
        boolean isManager = _contentWorkflowHelper.isAvailableAction((WorkflowAwareContent) content, 2);
        if (isManager)
        {
            content2Json.put("data", getContentData(content, searchUIModel));
            
            // user is allowed to edit content as producer => check model restriction on view items
            content2Json.put("notEditableDataIndex", canEdit(searchUIModel.getResultItems(Map.of()), content));
        }
        else
        {
            // Current path can be a partial path
            // Get full paths from current path: if unique compute rights from full path (better results), otherwise compute rights from partial path
            EducationalPath currentPath = EducationalPath.of(path.toArray(String[]::new));
            List<EducationalPath> fullPaths = _odfHelper.getEducationPathFromPath(currentPath);
            currentPath = fullPaths.size() == 1 ? fullPaths.getFirst() : currentPath;
            
            // Set current path in request attribute
            Request request = ContextHelper.getRequest(_context);
            request.setAttribute(ODFRightHelper.REQUEST_ATTR_EDUCATIONAL_PATHS, List.of(currentPath));
         
            boolean isConsumer = _contentWorkflowHelper.isAvailableAction((WorkflowAwareContent) content, EditContextualizedDataFunction.EDIT_WORKFLOW_ACTION_ID);
            
            content2Json.put("data", _getContentDataForPath(content, searchUIModel, EducationalPath.of(path.toArray(String[]::new)), isConsumer));
            
            if (isConsumer)
            {
                // user is allowed to edit content as consumer for current path => only repeater entries related to current path in tree can be edited
                content2Json.put("notEditableDataIndex", _canEditContextualizedDataPath(searchUIModel.getResultItems(Map.of()), content));
            }
            else
            {
                // workflow does not allow to edit content
                content2Json.put("notEditableData", true);
            }
        }
    }
    
    /**
     * List the path that cannot be edited.
     * Only attributes in a repeater with educational path could be modifiable according user's right and current path in tree.
     * @param viewItemAcessor The view with definition
     * @param content The content to study
     * @return The list of unmodifiable definition path
     */
    @SuppressWarnings("unchecked")
    protected List<String> _canEditContextualizedDataPath(ViewItemAccessor viewItemAcessor, Content content)
    {
        // Only attributes in a repeater with educational path could be modifiable
        
        List<String> results = new ArrayList<>();
        
        org.ametys.plugins.repository.model.ViewHelper.visitView(viewItemAcessor,
            (element, definition) -> {
                // attribute outside repeater with educational path is not contextualized and so not modifiable
                results.add(definition.getPath());
            },
            (group, definition) -> {
                // composite
                results.addAll(_canEditContextualizedDataPath(group, content));
            },
            (group, definition) -> {
                // repeater
                if (_educationalPathHelper.isRepeaterWithEducationalPath(definition))
                {
                    // Check if user is allowed to edit repeater data for current path
                    if (definition instanceof RestrictedModelItem restrictedModelItem && !restrictedModelItem.canWrite(content))
                    {
                        results.add(definition.getPath()); // user not allowed
                    }
                    else
                    {
                        // user is allowed on current path => check model restriction on child items
                        results.addAll(canEdit(group, content));
                    }
                }
                else
                {
                    // TODO call _canEditContextualizedDataPath if contains a repeater with educational path
                    results.add(definition.getPath());
                }
            },
            group -> {
                // group
                results.addAll(_canEditContextualizedDataPath(group, content));
            }
        );
        
        return results;
    }
    
    private Map<String, Object> _getContentDataForPath(Content content, SearchUIModel searchUIModel, EducationalPath currentPath, boolean isConsumer)
    {
        Map<String, Object> data = super.getContentData(content, searchUIModel);
        
        // Determines which entries are editable (depends on current path and user role)
        _checkMCCRepeater(content, data, "mccSession1", currentPath, isConsumer);
        _checkMCCRepeater(content, data, "mccSession2", currentPath, isConsumer);
        
        return data;
    }
    
    @SuppressWarnings("unchecked")
    private void _checkMCCRepeater(Content content, Map<String, Object> data, String repeaterName, EducationalPath currentPath, boolean isConsumer)
    {
        if (data.containsKey(repeaterName))
        {
            Map<String, Object> repeaterData = (Map<String, Object>) data.get(repeaterName);
            List<Map<String, Object>> entries = (List<Map<String, Object>>) repeaterData.get("entries");
            
            // browse repeater entries
            for (Map<String, Object> entry : entries)
            {
                if (!isConsumer)
                {
                    entry.put("notEditable", true);
                }
                else
                {
                    Map<String, Object> values = (Map<String, Object>) entry.get("values");
                    if (values != null)
                    {
                        boolean isCommon = (boolean) values.getOrDefault(EditContextualizedDataFunction.REPEATER_COMMON_ATTRIBUTE_NAME, true);
                        if (isCommon)
                        {
                            entry.put("notEditable", true); // a consumer cannot edit common MCC entries
                        }
                        else
                        {
                            // Compare educational path of entry with current path
                            ModelItem definition = content.getDefinition(repeaterName + "/" + EditContextualizedDataFunction.REPEATER_EDUCATIONAL_PATH_ATTRIBUTE_NAME);
                            if (definition.getType() instanceof EducationalPathRepositoryElementType pathType)
                            {
                                EducationalPath educationalPath = (EducationalPath) pathType.fromJSONForClient(values.get(EditContextualizedDataFunction.REPEATER_EDUCATIONAL_PATH_ATTRIBUTE_NAME), DataContext.newInstance());
                                entry.put("notEditable", !currentPath.equals(educationalPath));
                            }
                        }
                    }
                }
            }
        }
    }
    
    
    /**
     * Retrieve the course educational path from the parent program if there is only one parent
     * @param courseId The course id
     * @return The path from the parent program id of the course if the course is not shared or if all the paths lead to the same program, return null otherwise
     */
    @Callable (rights = Callable.READ_ACCESS, rightContext = ContentRightAssignmentContext.ID, paramIndex = 0)
    public String getCourseEducationalPath(String courseId)
    {
        Optional<String> programId = Optional.ofNullable(courseId)
            .filter(StringUtils::isNotBlank)
            .map(this::_resolveSilently)
            .filter(Course.class::isInstance)
            .map(Course.class::cast)
            .map(_odfHelper::getParentPrograms)
            .filter(parents -> parents.size() == 1)
            .map(Set::stream)
            .flatMap(Stream::findFirst)
            .map(Program::getId);
        
        if (programId.isPresent())
        {
            return EducationalPath.of(programId.get(), courseId).toString();
        }
        
        return null;
    }
    
    private <T> T _resolveSilently(String id)
    {
        try
        {
            return _ametysResolver.resolveById(id);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
}
