/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.property;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.model.properties.AbstractIndexableStaticProperty;
import org.ametys.cms.model.properties.Property;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.plugins.odfpilotage.helper.PilotageStatusHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageStatusHelper.PilotageStatus;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * {@link Property} for MCC workflow status on a {@link AbstractProgram}
 */
public class ProgramMCCWorkflowStatusProperty extends AbstractIndexableStaticProperty<String, String, AbstractProgram>
{
    private PilotageStatusHelper _pilotageStatusHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _pilotageStatusHelper = (PilotageStatusHelper) smanager.lookup(PilotageStatusHelper.ROLE);
    }
    
    public Object getValue(AbstractProgram program)
    {
        PilotageStatus pilotageStatus = _pilotageStatusHelper.getPilotageStatus(program);
        return pilotageStatus.name();
    }
    
    @Override
    public Enumerator<String> getEnumerator()
    {
        StaticEnumerator<String> enumerator = new StaticEnumerator<>();
        enumerator.addAll(Map.of(
                PilotageStatus.NONE.name(), new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_PROGRAM_PILOTAGE_STATUS_ENUM_NONE_LABEL"), // Add NONE value to the enumerator to use easily the search service in MCC dashboard tool
                PilotageStatus.MENTION_VALIDATED.name(), new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_PROGRAM_PILOTAGE_STATUS_ENUM_MENTION_LABEL"),
                PilotageStatus.ORGUNIT_VALIDATED.name(), new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_PROGRAM_PILOTAGE_STATUS_ENUM_ORGUNIT_LABEL"),
                PilotageStatus.CFVU_VALIDATED.name(), new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_PROGRAM_PILOTAGE_STATUS_ENUM_CFVU_LABEL")
        ));
        
        return enumerator;
    }

    @Override
    public boolean isMultiple()
    {
        return false;
    }
    
    @Override
    protected String getTypeId()
    {
        return ModelItemTypeConstants.STRING_TYPE_ID;
    }
}
