/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfpilotage.helper.ReportHelper;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;

/**
 * Pilotage report for Apogée.
 */
public class ApogeeReport extends AbstractReport
{
    private int _order;
    
    @Override
    protected String getType(Map<String, String> reportParameters)
    {
        return "apogee";
    }
    
    @Override
    public String getDefaultOutputFormat()
    {
        return OUTPUT_FORMAT_XLS;
    }

    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_XLS);
    }
    
    @Override
    public void _saxOrgUnit(ContentHandler handler, String catalog, String lang, String orgUnitId, Map<String, String> reportParameters)
    {
        OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
        
        try
        {
            handler.startDocument();
        
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("type", getType());
            XMLUtils.startElement(handler, "report", attrs);
            
            // SAX tree
            _generateReport(handler, orgUnit, lang, catalog);
            
            XMLUtils.endElement(handler, "report");
            handler.endDocument();
        }
        catch (Exception e)
        {
            getLogger().error("An error occured while generating 'Apogée' report for orgunit '{}'", orgUnit.getTitle(), e);
        }
    }
    
    /**
     * SAX the XML of the report
     * @param handler the handler
     * @param orgUnit the org unit
     * @param lang the lang of the programs
     * @param catalog the catalog of the programs
     * @throws SAXException if an error occurs while generating the SAX events
     */
    private void _generateReport(ContentHandler handler, OrgUnit orgUnit, String lang, String catalog) throws SAXException
    {
        _reportHelper.saxNaturesEnseignement(handler, getLogger());
        
        Map<Program, Object> contentsTree = _getStructure(orgUnit, lang, catalog);
        if (contentsTree.size() > 0)
        {
            _order = 1;
            _saxTree(handler, contentsTree);
        }
    }

    /**
     * Generate the data structure that will be used to create the report
     * @param rootOrgUnit the root org unit of the report
     * @param lang the lang of programs
     * @param catalog the catalog of programs
     * @return The structure
     */
    private Map<Program, Object> _getStructure(OrgUnit rootOrgUnit, String lang, String catalog)
    {
        Map<Program, Object> programTree = new TreeMap<>(ReportHelper.CONTENT_TITLE_COMPARATOR);
        
        // On ne récupère que la composante racine ou bien les composantes, enfant direct du root org unit, et on ignore les départements.
        if (rootOrgUnit.getParentOrgUnit() == null || rootOrgUnit.getParentOrgUnit().getParentOrgUnit() == null)
        {
            // Chercher les programmes concernés par la composante sélectionnée et ses enfants
            List<Program> programs = _odfHelper.getProgramsFromOrgUnit(rootOrgUnit, catalog, lang);
            for (Program program : programs)
            {
                Map<ProgramItem, Object> courses = _reportHelper.getCoursesFromContent(program);
                programTree.put(program, courses);
            }
        }
        
        return programTree;
    }

    /**
     * Sax the information related to the courses of the tree
     * @param handler the handler
     * @param programTree the program tree to sax
     * @throws SAXException if an error occurs when SAXing
     */
    @SuppressWarnings("unchecked")
    private void _saxTree(ContentHandler handler, Map<Program, Object> programTree) throws SAXException
    {
        for (Entry<Program, Object> programEntry : programTree.entrySet())
        {
            _saxCourseFromTree(handler, (Map<ProgramItem, Object>) programEntry.getValue(), programEntry.getKey());
        }
    }


    private void _saxCourseFromTree(ContentHandler handler, Map<ProgramItem, Object> programTree, Program program) throws SAXException
    {
        _saxCourseFromTree(handler, programTree, program, null, null, null, null, null, null, 1, "");
    }

    private void _saxCourseFromTree(ContentHandler handler, Map<ProgramItem, Object> tree, Program program, SubProgram subprogram, Container containerYear, Container containerSemester, CourseList list, Integer listPosition, Course parentCourse, int level, String courseHierarchy) throws SAXException
    {
        int courseListPosition = 0;
        for (Entry<ProgramItem, Object> entry : tree.entrySet())
        {
            ProgramItem child = entry.getKey();
            @SuppressWarnings("unchecked")
            Map<ProgramItem, Object> subTree = (Map<ProgramItem, Object>) entry.getValue();
            
            if (child instanceof Course childCourse)
            {
                String path = courseHierarchy +  " > " + childCourse.getTitle();
                _saxCourse(handler, program, subprogram, containerYear, containerSemester, list, listPosition, (Course) child, parentCourse, level, path);
                _saxCourseFromTree(handler, subTree, program, subprogram, containerYear, containerSemester, list, listPosition, childCourse, level + 1, path);
            }
            else if (child instanceof Program childProgram)
            {
                _saxCourseFromTree(handler, subTree, childProgram, subprogram, containerYear, containerSemester, list, listPosition, parentCourse, level, courseHierarchy);
            }
            else if (child instanceof Container childContainer)
            {
                String containerNature = _refTableHelper.getItemCode(childContainer.getNature());
                
                if ("annee".equals(containerNature))
                {
                    _saxCourseFromTree(handler, subTree, program, subprogram, childContainer, containerSemester, list, listPosition, parentCourse, level, courseHierarchy);
                }
                else if ("semestre".equals(containerNature))
                {
                    _saxCourseFromTree(handler, subTree, program, subprogram, containerYear, childContainer, list, listPosition, parentCourse, level, courseHierarchy);
                }
                else
                {
                    _saxCourseFromTree(handler, subTree, program, subprogram, containerYear, containerSemester, list, listPosition, parentCourse, level, courseHierarchy);
                }
            }
            else if (child instanceof SubProgram childSubProgram)
            {
                _saxCourseFromTree(handler, subTree, program, childSubProgram, containerYear, containerSemester, list, listPosition, parentCourse, level, courseHierarchy);
            }
            else if (child instanceof CourseList childCourseList)
            {
                courseListPosition++;
                String path = courseHierarchy.equals("") ? childCourseList.getTitle() : courseHierarchy +  " > " + childCourseList.getTitle();
                _saxCourseFromTree(handler, subTree, program, subprogram, containerYear, containerSemester, childCourseList, courseListPosition, parentCourse, level, path);
            }
        }
    }
    
    private String _getHierarchy(Program program, SubProgram subprogram, Container containerYear, Container containerSemester, String courseHierarchy)
    {
        String hierarchy = program.getTitle();
        if (subprogram != null)
        {
            hierarchy += " > " + subprogram.getTitle();
        }
        
        if (containerYear != null)
        {
            hierarchy += " > " + containerYear.getTitle();
        }
        
        if (containerSemester != null)
        {
            hierarchy += " > " + containerSemester.getTitle();
        }
            
        hierarchy += " > " + courseHierarchy;
        
        return hierarchy;
    }

    private void _saxCourse(ContentHandler handler, Program program, SubProgram subprogram, Container containerYear, Container containerSemester, CourseList list, Integer listPosition, Course course, Course parentCourse, int level, String courseHierarchy) throws SAXException
    {
        if (course != null)
        {
            String hierarchy = _getHierarchy(program, subprogram, containerYear, containerSemester, courseHierarchy);
            
            XMLUtils.startElement(handler, "course");
            
            // Ordre
            XMLUtils.createElement(handler, "ordre", String.valueOf(_order));
            
            // Composante
            _saxOrgUnits(handler, program);
            
            // Formation
            XMLUtils.createElement(handler, "formation", program.getTitle());
            XMLUtils.createElement(handler, "formationCode", program.getDisplayCode());
            
            _saxSubProgram(handler, subprogram);
            
            _saxContainer(handler, containerYear, parentCourse);
            
            _saxCourseList(handler, list, listPosition);
            
            // A des fils
            boolean aDesFils = course.hasCourseLists();
            XMLUtils.createElement(handler, "aDesFils", aDesFils ? "X" : "");

            long courseListsSize = course.getParentCourseLists().size();
            
            // Partagé
            XMLUtils.createElement(handler, "partage", courseListsSize > 1 ? "X" : "");

            // Nb occurrences
            XMLUtils.createElement(handler, "occurrences", _reportHelper.formatNumberToSax(courseListsSize));
            
            Container etape = _getEtapePorteuse(course, hierarchy);
            
            String porte = _getPorte(etape, containerYear);
           
            // Porté ("X" si l'ELP est porté par la formation (Etape porteuse=COD_ETP), vide sinon)
            XMLUtils.createElement(handler, "porte", porte);
            
            // Niveau
            XMLUtils.createElement(handler, "niveau", "niv" + level);

            // Date de création
            XMLUtils.createElement(handler, "creationDate", DateUtils.zonedDateTimeToString(course.getCreationDate()));

            // Code Apogée
            XMLUtils.createElement(handler, "codeApogee", course.getValue("elpCode", false, StringUtils.EMPTY));
            
            // Nature de l'élément
            XMLUtils.createElement(handler, "nature", _refTableHelper.getItemCode(course.getCourseType()));

            // Libellé court
            XMLUtils.createElement(handler, "libelleCourt", course.getValue("shortLabel", false, StringUtils.EMPTY));
            
            // Libellé
            XMLUtils.createElement(handler, "libelle", course.getTitle());
            
            // Code Ametys (ELP)
            XMLUtils.createElement(handler, "elpCode", course.getDisplayCode());
            
            // Lieu
            _reportHelper.saxContentAttribute(handler, course, "campus", "campus");

            // Crédits ECTS
            XMLUtils.createElement(handler, "ects", String.valueOf(course.getEcts()));
            
            String teachingActivity = _refTableHelper.getItemCode(course.getTeachingActivity());
            String stage = teachingActivity.equals("SA") ? "X" : "";
            
            // Element stage
            XMLUtils.createElement(handler, "stage", stage);
            
            // Code semestre et type de période (pair, impair, an)
            Content period = Optional.of("period")
                    .map(course::<ContentValue>getValue)
                    .flatMap(ContentValue::getContentIfExists)
                    .orElse(null);
            if (period != null)
            {
                try
                {
                    String periodCode = Optional.of("code")
                        .map(period::<String>getValue)
                        .orElse(StringUtils.EMPTY);
                    
                    String periodTypeCode = Optional.ofNullable(period.<ContentValue>getValue("type"))
                        .flatMap(ContentValue::getContentIfExists)
                        .map(c -> c.<String>getValue("code"))
                        .orElse(StringUtils.EMPTY);

                    XMLUtils.createElement(handler, "periode", "s10".equals(periodCode) ? "s0" : periodCode);
                    XMLUtils.createElement(handler, "periodeType", periodTypeCode);
                }
                catch (UnknownAmetysObjectException e)
                {
                    getLogger().error("Impossible de retrouver la période : {}", period, e);
                }
            }
            
            OrgUnit orgUnit = _getOrgUnit(course, hierarchy);
            
            // Code composante
            XMLUtils.createElement(handler, "codeComposante", orgUnit != null ? orgUnit.getValue("codCmp", false, StringUtils.EMPTY) : StringUtils.EMPTY);
            
            // Code CIP
            XMLUtils.createElement(handler, "codeCIP", orgUnit != null ? orgUnit.getValue("codCipApogee", false, StringUtils.EMPTY) : StringUtils.EMPTY);
            
            // Code ANU
            long codeAnu = course.getValue("CodeAnu", false, 0L);
            XMLUtils.createElement(handler, "CodeAnu", codeAnu > 0 ? String.valueOf(codeAnu) : StringUtils.EMPTY);
            
            // Calcul des charges
            XMLUtils.createElement(handler, "calculCharges", aDesFils ? "" : "X");
            
            // Heures d'enseignement
            for (CoursePart coursePart : course.getCourseParts())
            {
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("nature", coursePart.getNature());
                XMLUtils.createElement(handler, "volumeHoraire", attr, String.valueOf(coursePart.getNumberOfHours()));
            }

            // Etape porteuse pour les feuilles de l'arbre
            if (etape != null)
            {
                XMLUtils.createElement(handler, "etapePorteuse", etape.getValue("etpCode", false, ""));
                XMLUtils.createElement(handler, "vetEtapePorteuse", etape.getValue("vrsEtpCode", false, ""));
            }
            
            // Discipline
            String disciplineEnseignement = Optional.of("disciplineEnseignement")
                    .map(course::<ContentValue>getValue)
                    .flatMap(ContentValue::getContentIfExists)
                    .map(OdfReferenceTableEntry::new)
                    .map(entry -> {
                        String code = entry.getCode();
                        return (StringUtils.isNotEmpty(code) ? "[" + code + "] " : StringUtils.EMPTY) + entry.getLabel(course.getLanguage());
                    })
                    .orElse(StringUtils.EMPTY);
            XMLUtils.createElement(handler, "discipline", disciplineEnseignement);
            
            XMLUtils.endElement(handler, "course");
            
            _order++;
        }
    }
    
    private void _saxContainer(ContentHandler handler, Container container, Course parentCourse) throws AmetysRepositoryException, SAXException
    {
        if (container != null)
        {
            // Année
            XMLUtils.createElement(handler, "annee", container.getTitle());
            // COD_ETP
            XMLUtils.createElement(handler, "COD_ETP", container.getValue("etpCode", false, StringUtils.EMPTY));
            // COD_VRS_ETP
            XMLUtils.createElement(handler, "COD_VRS_ETP", container.getValue("vrsEtpCode", false, StringUtils.EMPTY));
            // Code Ametys ELP père
            XMLUtils.createElement(handler, "codeELPPere", parentCourse != null ? parentCourse.getCode() : "");
        }
    }
    
    private void _saxSubProgram(ContentHandler handler, SubProgram subprogram) throws AmetysRepositoryException, SAXException
    {
        if (subprogram != null)
        {
            // Parcours
            XMLUtils.createElement(handler, "parcours", subprogram.getTitle());
            XMLUtils.createElement(handler, "parcoursCode", subprogram.getDisplayCode());
        }
    }
    
    private void _saxOrgUnits(ContentHandler handler, Program program) throws SAXException
    {
        StringBuilder sb = new StringBuilder();
        List<String> orgUnits = program.getOrgUnits();
        for (String orgUnitId : orgUnits)
        {
            try
            {
                OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
                if (sb.length() > 0)
                {
                    sb.append(", ");
                }
                sb.append(orgUnit.getTitle());
                sb.append(" (");
                sb.append(orgUnit.getUAICode());
                sb.append(")");
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().info("La composante référencée par la formation {} ({}) n'a pas été trouvée.", program.getTitle(), program.getCode());
            }
        }
        XMLUtils.createElement(handler, "orgUnit", sb.toString());
    }
    
    private String _getPorte(Container etape, Container containerYear)
    {
        String porte = "";
        if (etape != null && containerYear != null)
        {
            String etpCode = containerYear.getValue("etpCode", false, StringUtils.EMPTY);
            if (StringUtils.isNotEmpty(etpCode))
            {
                porte = etpCode.equals(etape.getValue("etpCode", false, StringUtils.EMPTY)) ? "X" : StringUtils.EMPTY;
            }
        }
        return porte;
    }
    
    private OrgUnit _getOrgUnit(Course course, String hierarchy)
    {
        OrgUnit orgUnit = null;
        
        List<String> courseOrgUnits = course.getOrgUnits();
        if (!courseOrgUnits.isEmpty())
        {
            try
            {
                orgUnit = _resolver.resolveById(courseOrgUnits.get(0));
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().info("La composante référencée par l'élément pédagogique {} ({}) n'a pas été trouvée.", hierarchy, course.getCode());
            }
            
            if (courseOrgUnits.size() > 1)
            {
                getLogger().warn("L'élément pédagogique {} ({}) référence plus d'une composante.", hierarchy, course.getCode());
            }
        }
        
        return orgUnit;
    }
    
    private Container _getEtapePorteuse(Course course, String hierarchy)
    {
        return Optional.ofNullable((ContentValue) course.getValue("etapePorteuse"))
                .flatMap(contentValue -> _getEtapePorteuseIfExists(contentValue, course, hierarchy))
                .map(Container.class::cast)
                .orElse(null);
    }
    
    private Optional<ModifiableContent> _getEtapePorteuseIfExists(ContentValue etapePorteuse, Course course, String hierarchy)
    {
        try
        {
            return Optional.ofNullable(etapePorteuse.getContent());
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().info("L'année porteuse référencée par l'élément pédagogique {} ({}) n'a pas été trouvée.", hierarchy, course.getCode());
            return Optional.empty();
        }
    }
    
    private void _saxCourseList(ContentHandler handler, CourseList list, Integer position) throws SAXException
    {
        if (list != null)
        {
            XMLUtils.createElement(handler, "list", "Lst" + position);
            _saxChoiceList(handler, list);
        }
    }
    
    private void _saxChoiceList(ContentHandler handler, CourseList list) throws SAXException
    {
        // Type
        ChoiceType typeList = list.getType();
        if (typeList != null)
        {
            if (typeList.name() != null && (typeList.name().equals(ChoiceType.CHOICE.toString()) || typeList.name().equals(ChoiceType.MANDATORY.toString()) || typeList.name().equals(ChoiceType.OPTIONAL.toString())))
            {
                String typeListAsString = "";
                if (typeList.name().equals(ChoiceType.CHOICE.toString()))
                {
                    typeListAsString = "X";
                }
                else if (typeList.name().equals(ChoiceType.MANDATORY.toString()))
                {
                    typeListAsString = "O";
                }
                else if (typeList.name().equals(ChoiceType.OPTIONAL.toString()))
                {
                    typeListAsString = "F";
                }
                
                XMLUtils.createElement(handler, "typeList", typeListAsString);
            }
            
            // Min-Max (Ne remplir que pour le type "CHOICE" (ne mettre que le min))
            if (typeList.name() != null && typeList.name().equals(ChoiceType.CHOICE.toString()))
            {
                XMLUtils.createElement(handler, "minmax", _reportHelper.formatNumberToSax(list.getMinNumberOfCourses()));
            }
        }
    }
}
