/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.Binary;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.util.URIUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.Container;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.report.impl.tree.ProgramItemTree;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.rule.export.SetMCCHeaderAction;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.OrgUnitMCCReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.ProgramMCCReportSchedulable;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;

/**
 * Class to generate the MCC report.
 */
public class MCCReport extends AbstractMCCReport
{
    /** The key for the definitive versions */
    public static final String PARAMETER_DEFINITIVE = "definitive";
    
    /** The key for the alternatives versions */
    public static final String PARAMETER_RULES_VERSION = "rulesVersion";

    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _mccWorkflowHelper = (MCCWorkflowHelper) manager.lookup(MCCWorkflowHelper.ROLE);
    }
    
    @Override
    protected boolean isCompatibleSchedulable(AbstractReportSchedulable schedulable)
    {
        return schedulable instanceof ProgramMCCReportSchedulable || schedulable instanceof OrgUnitMCCReportSchedulable;
    }
    
    @Override
    protected boolean isSupportedTarget(PilotageReportTarget target)
    {
        return super.isSupportedTarget(target) || PilotageReportTarget.PROGRAM.equals(target);
    }
    
    @Override
    protected PilotageReportContent getReportContentForProgram(String outputFormat, Map<String, String> reportParameters)
    {
        return _getReportContentForProgram(outputFormat, reportParameters);
    }
    
    @Override
    protected String getType(Map<String, String> reportParameters)
    {
        String type = "mcc";
        
        if (reportParameters != null)
        {
            if (Boolean.parseBoolean(reportParameters.get(PARAMETER_DEFINITIVE)))
            {
                type += "-definitif";
            }
            else if (RulesManager.isRulesEnabled() && !reportParameters.get(PARAMETER_OUTPUT_FORMAT).equals(OUTPUT_FORMAT_XLS))
            {
                String rulesVersion = reportParameters.getOrDefault(PARAMETER_RULES_VERSION, "final");
                type += SetMCCHeaderAction.VERSION_TO_SUFFIX.getOrDefault(rulesVersion, "-" + rulesVersion);
            }
        }
        
        return type;
    }
    
    @Override
    protected boolean isExcludedFromMCC(ProgramItemTree tree)
    {
        return _pilotageHelper.isExcludedFromMCC((Course) tree.getCurrent());
    }
    
    @Override
    protected void saxSession(ContentHandler handler, ProgramItemTree tree, String sessionName) throws SAXException
    {
        Course course = (Course) tree.getCurrent();
        
        if (course.hasValue(sessionName) && !_disableConditionsEvaluator.evaluateDisableConditions(_sessionsView.getModelViewItem(sessionName).getDefinition(), sessionName, course))
        {
            XMLUtils.startElement(handler, sessionName);
            
            // Get all the full paths from the current educational path, this one can be partial.
            // For example, if have Year1;Semester2, we can obtain:
            //  - Program1;Year1;Semester2
            //  - Program2;SubProgram1;Year1;Semester2
            //  - Program2;SubProgram2;Year1;Semester2
            List<EducationalPath> educationalPaths = _odfHelper.getEducationPathFromPath(tree.getPath().getProgramItems(_resolver));
            
            // For each entry, check if the entry is common or the path correspond to one of the retrieved full educational paths
            for (ModelAwareRepeaterEntry sessionEntry : _odfHelper.getRepeaterEntriesByPath(course.getRepeater(sessionName), educationalPaths).toList())
            {
                _saxSessionEntry(handler, sessionEntry, course);
            }
            
            XMLUtils.endElement(handler, sessionName);
        }
    }
    
    /**
     * Sax a MCC session entry.
     * @param handler The handler
     * @param sessionEntry The session entry name
     * @throws SAXException If an error occurs
     */
    private void _saxSessionEntry(ContentHandler handler, ModelAwareRepeaterEntry sessionEntry, Content rootContent) throws SAXException
    {
        // start entry
        AttributesImpl entryAttrs = new AttributesImpl();
        entryAttrs.addCDATAAttribute("name", String.valueOf(sessionEntry.getPosition()));
        XMLUtils.startElement(handler, "entry", entryAttrs);
        
        saxSessionEntryDetails(handler, sessionEntry, rootContent);
        
        // end entry
        XMLUtils.endElement(handler, "entry");
    }
    
    @Override
    protected void populateMCCTree(ProgramItemTree tree)
    {
        ProgramItem programItem = tree.getCurrent();
        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
        for (ProgramItem child : children)
        {
            ProgramItemTree childTree = tree.addChild(child);
            populateMCCTree(childTree);
        }
    }
    
    @Override
    protected Function<ProgramItem, List<Pair<String, String>>> _getProgramItemTransform(String outputFormat, Map<String, String> reportParameters)
    {
        return Boolean.parseBoolean(reportParameters.get(PARAMETER_DEFINITIVE))
            // We want to get archived versions
            ? programItem -> {
                // The content have to be a container
                if (!(programItem instanceof Container container))
                {
                    throw new UnsupportedOperationException("The content '" + programItem.getId() + "' is not a Container and cannot be managed.");
                }
                
                ModelAwareRepeaterEntry entry = _mccWorkflowHelper.getLastMCCValidatedEntry(container);
                if (entry != null)
                {
                    String attributePath = entry.getHoldingRepeater().getModel().getName() + "[" + entry.getPosition() + "]/pdf";
                    String uri = attributePath + "?contentId=" + programItem.getId();
                    String fileUri = ResolveURIComponent.resolve("attribute", uri, false, false, true);
                    if (StringUtils.isNotEmpty(fileUri))
                    {
                        return List.of(Pair.of(entry.<Binary>getValue("pdf").getFilename(), fileUri));
                    }
                }
                
                // If there is no file, return null
                return null;
            }
            // Other cases
            : super._getProgramItemTransform(outputFormat, reportParameters);
    }
    
    @Override
    protected String _buildPipeline(String outputFormat, Map<String, String> reportParameters, String contentParameterName, AmetysObject content)
    {
        // If rules are enabled, generate the file with rules
        // Except for XLS format which is not supported with rules
        if (RulesManager.isRulesEnabled() && !outputFormat.equals(OUTPUT_FORMAT_XLS))
        {
            StringBuilder sb = new StringBuilder();
            sb.append("cocoon://_plugins/odf-pilotage/");
            sb.append(reportParameters.getOrDefault(PARAMETER_RULES_VERSION, "final"));
            sb.append("/container-rules.");
            sb.append(outputFormat);
            sb.append("?");
            sb.append(contentParameterName);
            sb.append("=");
            sb.append(URIUtils.encodeParameter(content.getId()));
            sb.append("&reportId=");
            sb.append(this.getId());
            return sb.toString();
        }
        
        // We want the "classic" MCC report file
        return super._buildPipeline(outputFormat, reportParameters, contentParameterName, content);
    }
    
    @Override
    protected void saxGlobalInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        super.saxGlobalInformations(handler, programItem, reportParameters);
        
        if (programItem instanceof Container container)
        {
            _reportHelper.saxMCCWorkflow(handler, container);
        }
    }
    
    @Override
    protected void saxOtherInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        super.saxOtherInformations(handler, programItem, reportParameters);
        
        if (ODFSkillsHelper.isSkillsEnabled() && !reportParameters.get(PARAMETER_OUTPUT_FORMAT).equals(OUTPUT_FORMAT_XLS))
        {
            _saxAcquiredMicroSkills(handler, programItem);
        }
    }
    
    private void _saxAcquiredMicroSkills(ContentHandler handler, ProgramItem programItem) throws SAXException
    {
        Map<Content, List<Content>> acquiredMicroSkillsByMacroSkills = _getAcquiredMicroSkills(programItem)
            .distinct()
            .map(ContentValue::getContentIfExists)
            .filter(Optional::isPresent)
            .map(Optional::get)
            .collect(Collectors.groupingBy(s -> s.<ContentValue>getValue("parentMacroSkill").getContent()));
        
        List<? extends Content> blockingMicroSkills = Stream.of(((Content) programItem).<ContentValue[]>getValue("parentPrograms/blockingMicroSkills", true))
            .distinct()
            .map(ContentValue::getContentIfExists)
            .filter(Optional::isPresent)
            .map(Optional::get)
            .toList();
        
        XMLUtils.startElement(handler, "skills");
        for (Content macroSkill : acquiredMicroSkillsByMacroSkills.keySet())
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", macroSkill.getId());
            XMLUtils.startElement(handler, "macro", attrs);
            
            XMLUtils.createElement(handler, "code", macroSkill.<String>getValue("skillCode"));
            XMLUtils.createElement(handler, "title", macroSkill.getTitle());
            
            for (Content microSkill : acquiredMicroSkillsByMacroSkills.get(macroSkill))
            {
                attrs.clear();
                attrs.addCDATAAttribute("id", microSkill.getId());
                attrs.addCDATAAttribute("isBlocking", Boolean.toString(blockingMicroSkills.contains(microSkill)));
                XMLUtils.startElement(handler, "micro", attrs);
                XMLUtils.createElement(handler, "code", microSkill.<String>getValue("skillCode"));
                XMLUtils.createElement(handler, "title", microSkill.getTitle());
                XMLUtils.endElement(handler, "micro");
            }
            
            XMLUtils.endElement(handler, "macro");
        }
        XMLUtils.endElement(handler, "skills");
    }
    
    private Stream<ContentValue> _getAcquiredMicroSkills(ProgramItem programItem)
    {
        Stream<ContentValue> acquiredMicroSkills = _odfHelper.getChildProgramItems(programItem).stream().flatMap(this::_getAcquiredMicroSkills);
        
        if (programItem instanceof Course course)
        {
            acquiredMicroSkills = Stream.concat(acquiredMicroSkills, Stream.of(course.<ContentValue[]>getValue("acquiredMicroSkills/microSkills", true)));
        }
        
        return acquiredMicroSkills;
    }
}
