/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.export.csv;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;

import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Context;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.RichTextHelper;
import org.ametys.cms.repository.Content;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.rule.ThematicsHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repositoryapp.RepositoryProvider;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Generator to export derogated or additional rules.
 */
public class ModifiedRulesExportGenerator extends ServiceableGenerator implements Contextualizable
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The thematics helper */
    protected ThematicsHelper _thematicsHelper;
    
    /** The rich text helper */
    protected RichTextHelper _richTextHelper;
    
    private Context _context;
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _thematicsHelper = (ThematicsHelper) smanager.lookup(ThematicsHelper.ROLE);
        _richTextHelper = (RichTextHelper) smanager.lookup(RichTextHelper.ROLE);
    }
    
    /**
     * Get the repository.
     * @return the repository.
     */
    public Repository getRepository()
    {
        return (Repository) _context.getAttribute(RepositoryProvider.CONTEXT_REPOSITORY_KEY);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "export");
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        String type = request.getParameter("type");
        String catalog = request.getParameter("catalog");
        String orgUnit = request.getParameter("orgUnit");
        
        _saxHeader(type);
        _saxRules(catalog, orgUnit, type);
        
        XMLUtils.endElement(contentHandler, "export");
        contentHandler.endDocument();
    }

    /**
     * Sax header column
     * @param type the type of rules (derogation or additional)
     * @throws SAXException if an error occurred
     */
    protected void _saxHeader(String type) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "header");
        
        new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_REGIME").toSAX(contentHandler, "column");
        new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_ID_THEMATIC").toSAX(contentHandler, "column");
        new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_NAME_THEMATIC").toSAX(contentHandler, "column");
        new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_ID_RULE").toSAX(contentHandler, "column");
        new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_ORGUNIT").toSAX(contentHandler, "column");
        new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_ID_CONTAINER").toSAX(contentHandler, "column");
        new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_TITLE_CONTAINER").toSAX(contentHandler, "column");
        if ("derogation".equals(type))
        {
            new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_DEROGATION").toSAX(contentHandler, "column");
        }
        else if ("complementary".equals(type))
        {
            new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_COMPLEMENTARY_RULE").toSAX(contentHandler, "column");
        }
        else
        {
            new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_ADDITIONAL_RULE").toSAX(contentHandler, "column");
        }
        new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_MOTIVATION").toSAX(contentHandler, "column");
        XMLUtils.endElement(contentHandler, "header");
    }
    
    /**
     * Sax the rules
     * @param catalog the catalog
     * @param orgUnit the orgUnit
     * @param type the type of rules (derogation or additional)
     * @throws SAXException if an error occurred
     */
    protected void _saxRules(String catalog, String orgUnit, String type) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "rules");
        Map<String, Content> thematics = _thematicsHelper.getThematics(catalog)
            .collect(
                Collectors.toMap(
                        c -> c.getValue(RulesManager.THEMATIC_CODE),
                        Function.identity()
                    )
                );
        
        Set<Container> containers = _getContainersWithModifiedRules(catalog, orgUnit, type);
        for (Container container : containers)
        {
            List<RuleRecord> rules = "complementary".equals(type)
                    ? _getComplementaryRuleRecords(container)
                    : _getRuleRecords(container, type, thematics);
            for (RuleRecord rule : rules)
            {
                _saxRule(container, rule);
            }
        }
        
        XMLUtils.endElement(contentHandler, "rules");
    }
    
    private List<RuleRecord> _getRuleRecords(Container container, String type, Map<String, Content> thematics)
    {
        Predicate<ModelAwareRepeaterEntry> filter = "derogation".equals(type)
                ? e -> !e.hasValue(RulesManager.RULE_THEMATIC_CODE)
                : e -> e.hasValue(RulesManager.RULE_THEMATIC_CODE);
        
        return Optional.ofNullable(container)
                .map(c -> c.<ModelAwareRepeater>getValue(RulesManager.CONTAINER_RULES))
                .map(ModelAwareRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .filter(filter)
                .map(e -> _entry2RuleRecord(e, thematics))
                .toList();
    }
    
    private List<RuleRecord> _getComplementaryRuleRecords(Container container)
    {
        List< ? extends ModelAwareRepeaterEntry> thematicEntries = Optional.ofNullable(container)
                .map(c -> c.<ModelAwareRepeater>getValue(RulesManager.CONTAINER_THEMATICS))
                .map(ModelAwareRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .toList();
        
        List<RuleRecord> rules = new ArrayList<>();
        for (ModelAwareRepeaterEntry thematicEntry : thematicEntries)
        {
            String thematicTitle = thematicEntry.getValue("title");
            String thematicCode = thematicEntry.getValue(RulesManager.THEMATIC_CODE);
            
            ModelAwareRepeater rulesRepeater = thematicEntry.getRepeater(RulesManager.THEMATIC_RULES);
            if (rulesRepeater != null)
            {
                for (ModelAwareRepeaterEntry entry : rulesRepeater.getEntries())
                {
                    rules.add(new RuleRecord(
                            entry,
                            null,
                            thematicCode,
                            thematicTitle
                        )
                    );
                }
            }
        }
        
        return rules;
    }
    
    private RuleRecord _entry2RuleRecord(ModelAwareRepeaterEntry entry, Map<String, Content> thematics)
    {
        String ruleCode = entry.getValue(RulesManager.RULE_CODE);
        String thematicCode = StringUtils.substringBefore(ruleCode, "-");
        Content thematic = thematics.get(thematicCode);
        
        return new RuleRecord(
            entry,
            thematic,
            thematicCode,
            thematic.getTitle()
        );
    }

    /**
     * Sax the rule
     * @param container the container holding the rule
     * @param rule the rule record
     * @throws SAXException if an error occurred
     */
    protected void _saxRule(Container container, RuleRecord rule) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "rule");
        ModelAwareRepeaterEntry entry = rule.ruleEntry();
        String ruleCode = entry.getValue(RulesManager.RULE_CODE);
        
        // Sax the thematic attributes
        _saxThematic(rule);
        
        // Sax the rule code
        _saxColumnValue("code", List.of(ruleCode));
        
        // Sax the program attributes
        _saxContainer(container);

        // Sax derogation
        RichText derogation = entry.getValue("text");
        List<String> derogationAsList = derogation != null ? List.of(_richTextHelper.richTextToString(derogation)) : List.of();
        _saxColumnValue("derogation", derogationAsList);
        
        // Sax motivation
        RichText motivation = entry.getValue("motivation");
        List<String> motivationAsList = motivation != null ? List.of(_richTextHelper.richTextToString(motivation)) : List.of();
        _saxColumnValue("motivation", motivationAsList);
        
        XMLUtils.endElement(contentHandler, "rule");
    }

    /**
     * Sax the thematic
     * @param thematic the thematic
     * @throws SAXException if an error occurred
     */
    protected void _saxThematic(RuleRecord thematic) throws SAXException
    {
        if (thematic != null)
        {
            Content content = thematic.thematicContent();
            List<String> regimes = Optional.ofNullable(content)
                    .map(c -> c.getValue("regime", false, new ContentValue[0]))
                    .map(Stream::of)
                    .orElseGet(Stream::of)
                    .map(ContentValue::getContentIfExists)
                    .filter(Optional::isPresent)
                    .map(Optional::get)
                    .map(c -> c.getTitle())
                    .toList();
            _saxColumnValue("regimes", regimes);
            _saxColumnValue("thematicCode", List.of(thematic.code()));
            _saxColumnValue("thematicTitle", List.of(thematic.title()));
        }
        else
        {
            _saxColumnValue("regimes", List.of());
            _saxColumnValue("thematicCode", List.of());
            _saxColumnValue("thematicTitle", List.of());
        }
    }
    
    /**
     * Sax the program attributes
     * @param container the container holding the rule
     * @throws SAXException if an error occurred
     */
    protected void _saxContainer(Container container) throws SAXException
    {
        List<String> orgUnits = container.getOrgUnits()
            .stream()
            .map(id -> _resolver.resolveById(id))
            .filter(OrgUnit.class::isInstance)
            .map(OrgUnit.class::cast)
            .map(OrgUnit::getTitle)
            .toList();
        
        _saxColumnValue("orgUnits", orgUnits);
        _saxColumnValue("containerCode", List.of(container.getDisplayCode()));
        _saxColumnValue("containerTitle", List.of(container.getTitle()));
    }

    /**
     * Get the containers with derogated or additional rules
     * @param catalog the catalog
     * @param orgUnit the orgUnit
     * @param type the type of rule to get (derogation or additional)
     * @return the set of containers
     */
    protected Set<Container> _getContainersWithModifiedRules(String catalog, String orgUnit, String type)
    {
        Set<Container> containers = new HashSet<>();
        
        StringBuilder queryString = new StringBuilder("//element(*, ametys:container)[");
        queryString.append(_getContainersPredicate(catalog, orgUnit));
        if ("complementary".equals(type))
        {
            queryString.append("]/ametys:thematics/*");
        }
        else
        {
            queryString.append("]/ametys:rules/*[");
            queryString.append(_getRulesPredicate(type));
            queryString.append("]");
        }
        
        try
        {
            Session session = getRepository().login();
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            @SuppressWarnings("deprecation")
            Query query = queryManager.createQuery(queryString.toString(), Query.XPATH);
            NodeIterator nodeIterator = query.execute().getNodes();
            while (nodeIterator.hasNext())
            {
                Node node = nodeIterator.nextNode();
                containers.add(_getParentContainer(node));
            }
        }
        catch (RepositoryException e)
        {
            getLogger().error("An error occurred getting containers with modified rules", e);
        }
        
        return containers;
    }

    private String _getContainersPredicate(String catalog, String orgUnit)
    {
        List<Expression> expressions = new ArrayList<>();
        expressions.add(new StringExpression("catalog", Operator.EQ, catalog));
        if (StringUtils.isNotBlank(orgUnit))
        {
            expressions.add(new StringExpression("orgUnit", Operator.EQ, orgUnit));
        }
        
        return new AndExpression(expressions).build();
    }
    
    private String _getRulesPredicate(String type)
    {
        if ("derogation".equals(type))
        {
            return new OrExpression(
                new NotExpression(new MetadataExpression(RulesManager.RULE_THEMATIC_CODE)),
                new StringExpression(RulesManager.RULE_THEMATIC_CODE, Operator.EQ, StringUtils.EMPTY)
            ).build();
        }
        else
        {
            // Additional rules have a thematic code
            return new StringExpression(RulesManager.RULE_THEMATIC_CODE, Operator.NE, StringUtils.EMPTY).build();
        }
    }
    
    private Container _getParentContainer(Node node)
    {
        try
        {
            Node parent = node.getParent();
            while (parent != null)
            {
                if (parent.getPrimaryNodeType().isNodeType(ContainerFactory.CONTAINER_NODETYPE))
                {
                    return _resolver.resolve(parent, false);
                }
                parent = parent.getParent();
            }
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred getting the parent container", e);
        }
        
        throw new AmetysRepositoryException("We should find a parent container from node : " + node);
    }
    
    private void _saxColumnValue(String attribute, List<String> values) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", attribute);
        XMLUtils.startElement(contentHandler, "column", attrs);
        for (String value : values)
        {
            XMLUtils.createElement(contentHandler, "value", value);
        }
        XMLUtils.endElement(contentHandler, "column");
    }
    
    private record RuleRecord(ModelAwareRepeaterEntry ruleEntry, Content thematicContent, String code, String title) { /* empty */ }
}
