/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Helper to open a dialog box to revert MCC state with comment
 * @private
 */
Ext.define('Ametys.plugins.odf.pilotage.helper.InvalidateDialog', {
	singleton: true,
	
    /**
     * Open dialog box comment field
     * @param {Object} config the dialog box configuration:
     * @param {String} config.dialogTitle The dialog title (required)
     * @param {String} config.dialogIconCls The css class for dialog icon (required)
	 * @param {String} [config.hintMsg] A optional hint message
     * @param {Function} config.revertCallback The callback function (required). The callback will be called with following arguments:
     * @param {String} config.revertCallback.comment The comment value
	 * @param {Boolean} [autoCloseDialog=true] Set to false to not close dialog after clicking on validate button
     * @return the dialog box
     */
	open: function (config)
    {
		config = Ext.applyIf(config, {
            autoCloseDialog: true
        });
				
        this._createDialog(config);
        
        this._dialog.show();
        
        return this._dialog;
    },
    
    _createFormPanel: function (config)
    {
		var me = this;
        return Ext.create('Ext.form.Panel', {
            defaults: {
                cls: 'ametys',
                labelSeparator: '',
                labelAlign: 'top',
                width: '100%',
                msgTarget: 'side'
            },
            
            border: false,
            scrollable: false,
            
            items: [
				{
					xtype: 'radiofield',
                    name: 'reset',
                    inputValue: true,	
					checked: true,				
					boxLabel: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_RESET_MCC_WORKFLOW_LABEL}}",
					ametysDescription: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_RESET_MCC_WORKFLOW_DESC}}",
					hidden: config.noReset == true,
                }, 
				{
					xtype: 'radiofield',
                    name: 'reset',
                    inputValue: false,					
					boxLabel: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_UNDO_MCC_WORKFLOW_LABEL}}",
					ametysDescription: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_UNDO_MCC_WORKFLOW_DESC}}",
					hidden: config.noReset == true,
                },			
				{
                    xtype: 'checkbox',
                    name: 'sendMail',
                    itemId: 'sendMail',
					inputValue: true,
					uncheckedValue: false,
					checked: true,
                    boxLabel: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_SENDMAIL_LABEL}}",
                    ametysDescription: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_SENDMAIL_DESC}}",
					listeners: {
						'change': function(field, newValue, oldValue) {
							me._form.getForm().findField('comment').setDisabled(!newValue);
						}
					}
                },
                {
                    xtype: 'textareafield',
                    name: 'comment',
                    itemId: 'comment',
					height: 150,
                    fieldLabel: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_COMMENT_LABEL}}",
                    ametysDescription: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_COMMENT_DESC}}"
                }
            ]
        });
    },
    
    _createDialog: function(config)
    {
        this._form = this._createFormPanel(config);
        
        var items = [];
        if (config.hintMsg)
        {
            items.push({
                xtype: 'component',
                cls: 'a-text',
                html: config.hintMsg
            })
        }
        items.push(this._form);
        
        this._dialog = Ext.create('Ametys.window.DialogBox', {
            title: config.dialogTitle,
            iconCls: config.dialogIconCls,
            
            width: 450, 
            scrollable: false,
            
            items: items,
            
            referenceHolder: true,
            defaultButton: 'validate',
            closeAction: 'destroy',
            
            buttons: [{
                reference: 'validate',
                text: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_OK_BTN}}",
                handler: Ext.bind(this._validateFn, this, [config]),
                scope: this
            }, {
                text: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_HELPER_INVALIDATION_DIALOG_CANCEL_BTN}}",
                handler : function () { this._dialog.close()},
                scope: this
            }]    
        });
        
        return this._dialog;
    },
    
    _validateFn: function (config)
    {
        var form = this._form.getForm();
        if (!form.isValid())
        {
            return;
        }
        
        var values = form.getValues();
        
        config.revertCallback(values.sendMail, values.comment, config.noReset ? false : values.reset);
        
        if (config.autoCloseDialog)
        {
            this._dialog.close();
        }
    }
});
