/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * This class controls a ribbon button that allows to set the MCC excluded state of a ELP.<br/>
 * The button is toggle if content is currently excluded from MCC.
 * @private
 */
Ext.define('Ametys.plugins.odf.pilotage.controller.ExcludeCourseFromMCCController', {
	extend: 'Ametys.plugins.cms.content.controller.SmartContentController',
		
	/**
	 * @private
	 * @property {String[]} _excludedELPIds The ids of excluded ELPs from MCC among the current selection
	 */
	/**
	 * @private
	 * @property {String[]} _includedELPIds The ids of included EPLs in MCC among the current selection
	 */
	
	/**
	 * @cfg {String} [icon-glyph] The CSS class for button's icon when the ELP is included in MCC
	 */
    /**
     * @cfg {String} [excluded-icon-glyph] The CSS class for button's icon when the ELP is excluded from MCC
     */
	
	constructor: function(config)
	{
		this.callParent(arguments);
		
		Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
	},
	
	/**
	 * Listener handler for modified messages
	 * @param {Ametys.message.Message} message the message
	 */
	 _onModified: function(message)
	 { 
		 if (this.updateTargetsInCurrentSelectionTargets (message))
		 {
			 this.refresh();
		 }
	 },
	 
	/**
	 * Returns the ids of excluded ELPs from MCC among the current selection
	 * @return {String[]} The contents' id
	 */
	getExcludedELPIds: function ()
	{
		return this._excludedELPIds;
	},
	
	/**
	 * Returns the ids of included ELPs from MCC among the current selection
	 * @return {String[]} The contents' id
	 */
	getIncludedELPIds: function ()
	{
		return this._includedELPIds;
	},
	
	_getStatus: function (targets)
	{
		this._contentIds = [];
		this._excludedELPIds = [];
		this._includedELPIds = [];
		
        this.disable();
        this.callParent(arguments);
	},

    _calculateStatus: function(targets)
    {
        var calculateStatus = this.callParent(arguments);
        
        calculateStatus["excluded-contents"] = [];
        calculateStatus["included-contents"] = [];

        Ext.Array.each(targets, function(contentTarget)
        {
            var parameters = contentTarget.getParameters();
            if (parameters && parameters.content)
            {
                var content = parameters.content;
                if (parameters.additionalData.isExcludedFromMCC)
                {
                    var i18nStr = this.getConfig("excluded-contents-description");
                    var description = Ext.String.format(i18nStr, content.getTitle());
                    var contentParam = this._getContentDefaultParameters(content);
                    contentParam["description"] = description;
                    calculateStatus["excluded-contents"].push(contentParam);
                }
                else
                {
                    var i18nStr = this.getConfig("included-contents-description");
                    var description = Ext.String.format(i18nStr, content.getTitle());
                    var contentParam = this._getContentDefaultParameters(content);
                    contentParam["description"] = description;
                    calculateStatus["included-contents"].push(contentParam);
                }
                
            }
        }, this);
        return calculateStatus;
    },
	
	_getStatusCb: function(params)
	{
        this.callParent(arguments);
        
		var me = this;
        Ext.Array.each (params['excluded-contents'], function (content) {
            me._excludedELPIds.push(content.id);
		});
		
        Ext.Array.each (params['included-contents'], function (content) {
            me._includedELPIds.push(content.id);
		});

        this.toggle(this._excludedELPIds.length > 0);
		this._updateIcons(params);
	},
	
	_updateTooltipDescription: function (description, params)
	{
        this.callParent(arguments);
        
        var parentDescription = this._description;
        
		parentDescription = this._handlingMultiple (parentDescription, 'included-contents', params['included-contents']);
		parentDescription = this._handlingMultiple(parentDescription, "excluded-contents", params['excluded-contents']);
		
		this.setDescription (parentDescription);
	},
	
	/**
     * @private
	 * Update the icons of the button
	 * @param {Object} params the server's parameters
	 */
	_updateIcons: function(params)
	{
		var nbIncludedContents = params['included-contents'].length;
		var nbExcludedContents = params['excluded-contents'].length;
		
		var iconGlyph = this.getInitialConfig()['icon-glyph'];
		if (nbExcludedContents > 0)
		{
			iconGlyph = this.getInitialConfig()['excluded-icon-glyph'];
		}
        
        this.setGlyphIcon(iconGlyph);
	}
    
});