/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.actions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.right.RightManager.RightResult;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.AbstractWorkflowHelper.ODFWorkflowStep;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowStep;
import org.ametys.plugins.odfpilotage.rule.RulesManager;

/**
 * Actions to get the MCC workflow of a {@link ProgramItem}
 *
 */
public class MCCWorkflowHistoryAction extends AbstractProgramItemHistoryAction
{
    private MCCWorkflowHelper _mccWorkflowHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _mccWorkflowHelper = (MCCWorkflowHelper) smanager.lookup(MCCWorkflowHelper.ROLE);
    }
    
    @Override
    protected boolean canSeeHistory(ProgramItem programItem)
    {
        return _rightManager.currentUserHasRight("ODF_Pilotage_See_MCCHistory_Rights", programItem) == RightResult.RIGHT_ALLOW;
    }

    @Override
    protected ProgramItem getParentProgramItemForHistory(ProgramItem programItem)
    {
        return _mccWorkflowHelper.getParentContainerWithHigherMCCStatus(programItem);
    }

    @Override
    protected List<ODFWorkflowStep> getWorkflowSteps(ProgramItem programItem)
    {
        List<ODFWorkflowStep> workflowSteps = new ArrayList<>();
        
        if (programItem instanceof Container container)
        {
            if (RulesManager.isRulesEnabled() && _mccWorkflowHelper.isRulesValidated(container))
            {
                workflowSteps.add(_mccWorkflowHelper.getRulesMentionValidationStep(container));
            }
            
            if (_mccWorkflowHelper.isMCCValidated(container))
            {
                workflowSteps.add(_mccWorkflowHelper.getMCCMentionValidationStep(container));
            }
            
            if (_mccWorkflowHelper.isMCCOrgUnitControlled(container))
            {
                workflowSteps.add(_mccWorkflowHelper.getMCCOrgunitControlStep(container));
            }
            
            if (_mccWorkflowHelper.isMCCOrgUnitValidated(container))
            {
                workflowSteps.add(_mccWorkflowHelper.getMCCOrgunitValidationStep(container));
            }
            
            if (_mccWorkflowHelper.isMCCCFVUControlled(container))
            {
                workflowSteps.add(_mccWorkflowHelper.getMCCCFVUControlStep(container));
            }
            
            if (_mccWorkflowHelper.isMCCCFVUValidated(container))
            {
                workflowSteps.add(_mccWorkflowHelper.getMCCCFVUValidationStep(container));
            }
            
            // Order step by date
            Collections.sort(workflowSteps, new MCCStepComparator());
        }
        
        return workflowSteps;
    }
    
    private static final class MCCStepComparator implements Comparator<ODFWorkflowStep>
    {
        public int compare(ODFWorkflowStep s1, ODFWorkflowStep s2)
        {
            if (MCCWorkflowStep.MCC_CFVU_VALIDATED.name().equals(s1.id()))
            {
                return -1;
            }
            
            if (MCCWorkflowStep.MCC_CFVU_VALIDATED.name().equals(s2.id()))
            {
                return 1;
            }
            
            if (MCCWorkflowStep.MCC_CFVU_CONTROLLED.name().equals(s1.id()))
            {
                return -1;
            }
            
            if (MCCWorkflowStep.MCC_CFVU_CONTROLLED.name().equals(s2.id()))
            {
                return 1;
            }
            
            if (MCCWorkflowStep.MCC_ORGUNIT_VALIDATED.name().equals(s1.id()))
            {
                return -1;
            }
            
            if (MCCWorkflowStep.MCC_ORGUNIT_VALIDATED.name().equals(s2.id()))
            {
                return 1;
            }
            
            if (MCCWorkflowStep.MCC_ORGUNIT_CONTROLLED.name().equals(s1.id()))
            {
                return -1;
            }
            
            if (MCCWorkflowStep.MCC_ORGUNIT_CONTROLLED.name().equals(s2.id()))
            {
                return 1;
            }
            
            return s1.date().compareTo(s2.date());
        }
    }
}
