/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * This controller is used for the steering buttons of MCC workflow
 */
Ext.define('Ametys.plugins.odf.pilotage.controller.MCCWorkflowController', {
    extend: 'Ametys.plugins.cms.content.controller.SmartContentController',
    
    /**
     * @property
     * @type {Number} MAX_CONTENTS_IN_TOOLTIP
     */
    MAX_CONTENTS_IN_TOOLTIP: 5,
    
    /**
     * @cfg {String} mcc-workflow-action The MCC workflow action of this controller
     */
    /**
     * @property {String} _mccWorkflowAction See #cfg-workflow-action
     * @private
     */
    
    constructor: function(config)
    {
        this.callParent(arguments);
        
        this._mccWorkflowAction = this.getInitialConfig("mcc-workflow-action");
		
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
    },
    
    /**
     * @private
     * Listener handler for modified messages
     * @param {Ametys.message.Message} message the message
     */
    _onModified: function(message)
    {
        if (this.updateTargetsInCurrentSelectionTargets(message))
        {
            this.refresh();
        }
    },
    
    /**
	 * Refresh the controller from the content informations state of given targets
	 * @param targets The content targets
	 * @protected
	 */
	_getStatus: function (targets)
	{
		this.disable();
		
		var contentIds = [];
		for (var i=0; i < targets.length; i++)
		{
			contentIds.push(targets[i].getParameters().id);
		}
        
		this.serverCall ('getStatus', [contentIds, this._mccWorkflowAction], this._getStatusCb, { arguments: targets, cancelCode: this.self.getName() + "$" + this.getId(), refreshing: true });
	},
    
    /**
     * @protected
     * Update the tooltip description according state of the current selection
     * @param description The initial description. Can be empty.
     * @param params The JSON result received
     */
    _updateTooltipDescription: function (description, params)
    {
        description = this._handlingMultiple(description, "allright", params['allright-contents']);
		description = this._handlingMultiple(description, "invalidmccstatus", params['invalidmccstatus-contents']);
        description = this._handlingMultiple(description, "noyear", params['noyear-contents']);
        description = this._handlingMultiple(description, "noright", params['noright-contents']);
		description = this._handlingMultiple(description, "noread", params['noread-contents']);
        description = this._handlingMultiple(description, "locked", params['locked-contents']);
        
        this.setDescription (description);
    },
    
    // Override to add "... and X others"
    _handlingMultiple: function(description, prefix, contents)
    {
        if (contents && contents.length > 0)
        {
            if (description != "")
            {
                description += "<br/><br/>";
            }
            
            description += this.getInitialConfig(prefix + "-start-description");
            for (var i=0; i < Math.min(this.MAX_CONTENTS_IN_TOOLTIP, contents.length); i++)
            {
                if (i != 0) 
                {
                    description += ", ";
                }
                description += contents[i].description;
            }
            
            let remainingContents = contents.length - this.MAX_CONTENTS_IN_TOOLTIP;
            if (remainingContents == 1)
            {
                description += "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_TOOLTIP_REMAINING_CONTENT}}";
            }
            else if (remainingContents > 1)
            {
                description += Ext.String.format("{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_TOOLTIP_REMAINING_CONTENTS}}", remainingContents);
            }
            
            description += this.getInitialConfig(prefix + "-end-description");
        }
        
        return description;
    }
});
