/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.export.csv;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.util.I18nUtils;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.plugins.odfpilotage.rule.export.AbstractSetContentDispositionHeaderAction;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * This action set the modified rules csv name as attachment in the header
 */
public class SetModifiedRulesHeaderCSVAction extends AbstractSetContentDispositionHeaderAction implements Serviceable
{
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The I18n utils */
    protected I18nUtils _i18nUtils;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    protected String buildFilename(Request request, Parameters parameters) throws Exception
    {
        String lang = Config.getInstance().getValue("odf.programs.lang");
        String type = parameters.getParameter("type");
        String catalog = parameters.getParameter("catalog");
        String orgUnitId = parameters.getParameter("orgUnit");
        OrgUnit orgUnit = StringUtils.isNotBlank(orgUnitId) ? _resolver.resolveById(orgUnitId) : null;
        
        StringBuilder filename = new StringBuilder();
        switch (type)
        {
            case "derogation":
                filename.append(_i18nUtils.translate(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_DEROGATIONS_FILE_TITLE"), lang));
                break;
            case "complementary":
                filename.append(_i18nUtils.translate(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_COMPLEMENTARY_FILE_TITLE"), lang));
                break;
            default:
                filename.append(_i18nUtils.translate(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_DASHBOARD_EXPORT_CSV_ADDITIONAL_FILE_TITLE"), lang));
        }
        filename.append(" - ");
        
        filename.append(_catalogsManager.getCatalog(catalog).getTitle());

        if (orgUnit != null)
        {
            filename.append(" - ");
            filename.append(orgUnit.getTitle());
        }
        filename.append(".csv");
        
        return filename.toString();
    }
}
